searchData={"items":[{"type":"module","title":"dbg","doc":"The Text Based Trace Facility\n\nThis module implements a text based interface to the\n`trace:process/4`, `trace:port/4`, and `trace:function/4` BIFs,\nsimplifying tracing of functions, processes, ports, and messages.\n\nTo quickly get started on tracing function calls you can use the\nfollowing code in the Erlang shell:\n\n```erlang\n1> dbg:tracer().  % Start the default trace message receiver\n{ok,<0.90.0>}\n2> dbg:p(all, c). % Set upp call tracing on all processes\n{ok,[{matched,nonode@nohost,49}]}\n3> dbg:tp(lists, seq, cx). %  Set up call and exception tracing on lists:seq/2,3\n{ok,[{matched,nonode@nohost,2},{saved,cx}]}\n4> lists:seq(1, 10).\n(<0.88.0>) call lists:seq(1,10) ({erl_eval,do_apply,7,{\"erl_eval.erl\",904}})\n[1,2,3,4,5,6,7,8,9,10]\n(<0.88.0>) returned from lists:seq/2 -> [1,2,3,4,5,6,7,8,9,10]\n```\n\nThe utilities are also suitable to use in system testing on large systems, where\nother tools have too severe impact on the system performance. Some primitive\nsupport for sequential tracing is also included; see the\n[advanced topics](`m:dbg#advanced`) section.","ref":"dbg.html"},{"type":"module","title":"Simple tracing from the shell with no prior set up - dbg","doc":"To trace a call to a function with minimal fuss, call [`dbg:c(Module,\nName, Arguments)`](`dbg:c/3`). `dbg:c/3` starts a temporary trace\nreceiver, enables all trace flags, and calls the designated function\nfrom a temporary process. For example, here is how to trace a call\nto `application:which_applications/0`:\n\n```erlang\n1> dbg:c(application, which_applications, []).\n(<0.92.0>) <0.45.0> ! {'$gen_call',{<0.92.0>,\n                                    [alias|\n                                     #Ref<0.0.11779.270031856.1478295555.230456>]},\n                                   which_applications} (Timestamp: {1710,\n                                                                    847802,\n                                                                    479222})\n(<0.92.0>) out {gen,do_call,4} (Timestamp: {1710,847802,479231})\n(<0.92.0>) in {gen,do_call,4} (Timestamp: {1710,847802,479271})\n(<0.92.0>) << {[alias|#Ref<0.0.11779.270031856.1478295555.230456>],\n               [{stdlib,\"ERTS  CXC 138 10\",\"5.2.1\"},\n                {kernel,\"ERTS  CXC 138 10\",\"9.2.2\"}]} (Timestamp: {1710,\n                                                                   847802,\n                                                                   479274})\n[{stdlib,\"ERTS  CXC 138 10\",\"5.2.1\"},\n {kernel,\"ERTS  CXC 138 10\",\"9.2.2\"}]\n```\n\nFour trace events are generated:\n\n- A send event (`!`) for the sending of a request from the current process\n  to the `application_controller` process.\n- A schedule-out event (`out`) when the current process schedules out while\n  waiting in a `receive` for the reply to arrive.\n- A schedule-in event (`in`) when the current process is scheduled in when\n  reply has arrived.\n- A `receive` event (`<<`) when the current process retrieves the reply from\n  the `application_controller` process.\n\nThe `dbg:c/4` function has a fourth argument for specifying the trace flags.\nHere is how to only show message sending and receiving:\n\n```erlang\n2> dbg:c(application, which_applications, [], m).\n(<0.96.0>) <0.45.0> ! {'$gen_call',{<0.96.0>,\n                                    [alias|\n                                     #Ref<0.0.12291.270031856.1478295555.230496>]},\n                                   which_applications}\n(<0.96.0>) << {[alias|#Ref<0.0.12291.270031856.1478295555.230496>],\n               [{stdlib,\"ERTS  CXC 138 10\",\"5.2.1\"},\n                {kernel,\"ERTS  CXC 138 10\",\"9.2.2\"}]}\n[{stdlib,\"ERTS  CXC 138 10\",\"5.2.1\"},\n {kernel,\"ERTS  CXC 138 10\",\"9.2.2\"}]\n```","ref":"dbg.html#module-simple-tracing-from-the-shell-with-no-prior-set-up"},{"type":"module","title":"Tracing from the shell - dbg","doc":"Another way of tracing from the shell is to explicitly start a _tracer_ and\nset the _trace flags_ of your choice on the processes you want to trace.\nFor example, here is how to trace messages and process events:\n\n```erlang\n1> Pid = spawn(fun() -> receive {From,Msg} -> From ! Msg end end).\n<0.90.0>\n2> dbg:tracer().\n{ok,<0.92.0>}\n3> dbg:p(Pid, [m,procs]).\n{ok,[{matched,nonode@nohost,1}]}\n4> Pid ! {self(),hello}.\n(<0.90.0>) << {<0.88.0>,hello}\n{<0.88.0>,hello}\n(<0.90.0>) <0.88.0> ! hello\n(<0.90.0>) exit normal\n5> flush().\nShell got hello\nok\n```\n\nIn order to trace functions call, in addition to enabling the `call` trace flag\nfor the process, it is also necessary to set a _trace pattern_ for the functions\nto trace.\n\n_Example:_\n\n```erlang\n1> dbg:tracer().\n{ok,<0.90.0>}\n2> dbg:p(all, call).\n{ok,[{matched,nonode@nohost,49}]}\n3> dbg:tp(lists, last, 1, []).\n{ok,[{matched,nonode@nohost,1}]}\n4> lists:last([a,b,c,d,e]).\n(<0.88.0>) call lists:last([a,b,c,d,e])\ne\n5> dbg:tp(lists, last, 1, [{'_',[],[{return_trace}]}]).\n{ok,[{matched,nonode@nohost,1},{saved,1}]}\n6> lists:last([a,b,c,d,e]).\n(<0.88.0>) call lists:last([a,b,c,d,e])\n(<0.88.0>) returned from lists:last/1 -> e\ne\n```\n\n[](){: #advanced }","ref":"dbg.html#module-tracing-from-the-shell"},{"type":"module","title":"Advanced topics - combining with seq_trace - dbg","doc":"The `dbg` module is primarily targeted towards tracing through the\n`trace:process/4` function. It is sometimes desired to trace messages in a more\ndelicate way, which can be done with the help of the `m:seq_trace` module.\n\n`m:seq_trace` implements sequential tracing (known in the AXE10 world, and\nsometimes called \"forlopp tracing\"). `dbg` can interpret messages generated from\n`seq_trace` and the same tracer function for both types of tracing can be used.\nThe `seq_trace` messages can also be sent to a trace port for further analysis.\n\nAs a match specification can turn on sequential tracing, the combination of\n`dbg` and `seq_trace` can be powerful. This brief example shows a session\nwhere sequential tracing is used to trace the `dbg` module and the trace itself:\n\n```erlang\n1> dbg:tracer().\n{ok,<0.30.0>}\n2> {ok, Tracer} = dbg:get_tracer().\n{ok,<0.31.0>}\n3> seq_trace:set_system_tracer(Tracer).\nfalse\n4> dbg:tp(dbg, get_tracer, 0, [{[],[],[{set_seq_token, send, true}]}]).\n{ok,[{matched,nonode@nohost,1},{saved,1}]}\n5> dbg:p(all,call).\n{ok,[{matched,nonode@nohost,22}]}\n6> dbg:get_tracer(), seq_trace:set_token([]).\n(<0.25.0>) call dbg:get_tracer()\nSeqTrace [0]: (<0.25.0>) <0.30.0> ! {<0.25.0>,get_tracer} [Serial: {2,4}]\nSeqTrace [0]: (<0.30.0>) <0.25.0> ! {dbg,{ok,<0.31.0>}} [Serial: {4,5}]\n{1,0,5,<0.30.0>,4}\n```\n\nThis session sets the system_tracer to the same process as the\nordinary tracer process (i. e. <0.31.0>) and sets the trace pattern\nfor the function `dbg:get_tracer` to one that has the action of\nsetting a sequential token. When the function is called by a traced\nprocess (all processes are traced in this case), the process gets\n\"contaminated\" by the token and `seq_trace` messages are sent both for\nthe server request and the response. The `seq_trace:set_token([])`\nafter the call clears the `seq_trace` token, which is why no messages\nare sent when the answer propagates via the shell to the console\nport. Otherwise the output would been more noisy.","ref":"dbg.html#module-advanced-topics-combining-with-seq_trace"},{"type":"module","title":"Note of caution - dbg","doc":"When tracing function calls on a group leader process (an I/O process), there is\nrisk of causing a deadlock. This will happen if a group leader process generates\na trace message and the tracer process, by calling the trace handler function,\nsends an I/O request to the same group leader. The problem can only occur if the\ntrace handler prints to the tty using an `m:io` function such as\n[`format/2`](`io:format/2`). Note that when `dbg:p(all, call)` is called, IO\nprocesses are also traced. Here is an example:\n\n```erlang\n%% Using a default line editing shell\n1> dbg:tracer(process, {fun(Msg,_) -> io:format(\"~p~n\", [Msg]), 0 end, 0}).\n{ok,<0.37.0>}\n2> dbg:p(all, [call]).\n{ok,[{matched,nonode@nohost,25}]}\n3> dbg:tp(mymod,[{'_',[],[]}]).\n{ok,[{matched,nonode@nohost,0},{saved,1}]}\n4> mymod: % TAB pressed here\n%% -- Deadlock --\n```\n\nHere is another example:\n\n```erlang\n%% Using a shell without line editing (oldshell)\n1> dbg:tracer(process).\n{ok,<0.31.0>}\n2> dbg:p(all, [call]).\n{ok,[{matched,nonode@nohost,25}]}\n3> dbg:tp(lists,[{'_',[],[]}]).\n{ok,[{matched,nonode@nohost,0},{saved,1}]}\n% -- Deadlock --\n```\n\nThe reason we get a deadlock in the first example is because when TAB is pressed\nto expand the function name, the group leader (which handles character input)\ncalls `mymod:module_info()`. This generates a trace message which, in turn,\ncauses the tracer process to send an IO request to the group leader (by calling\n`io:format/2`). We end up in a deadlock.\n\nIn the second example we use the default trace handler function. This\nhandler prints to the tty by sending IO requests to the `user`\nprocess. When Erlang is started in the oldshell mode, the shell\nprocess will have `user` as its group leader and so will the tracer\nprocess in this example. Since `user` calls functions in `lists` we\nend up in a deadlock as soon as the first IO request is sent.\n\nHere are a few suggestions for avoiding deadlock:\n\n- Do not trace the group leader of the tracer process. If tracing has been\n  switched on for all processes, call `dbg:p(TracerGLPid, clear)` to stop tracing\n  the group leader (`TracerGLPid`).\n  [`process_info(TracerPid, group_leader)`](`process_info/2`) tells you which\n  process this is (`TracerPid` is returned from `dbg:get_tracer/0`).\n- Do not trace the `user` process if using the default trace handler function.\n- In your own trace handler function, call `erlang:display/1` instead of an `io`\n  function or, if `user` is not used as group leader, print to `user` instead of\n  the default group leader. Example: `io:format(user, Str, Args)`.","ref":"dbg.html#module-note-of-caution"},{"type":"function","title":"dbg.c/3","doc":"","ref":"dbg.html#c/3"},{"type":"function","title":"dbg.c/4","doc":"Evaluates the expression [`apply(Mod, Fun, Args)`](`apply/3`) with the\ntrace flags in `Flags` set.\n\n`c` stands for **c**all.\n\nThis is a convenient way to trace processes from the Erlang shell.","ref":"dbg.html#c/4"},{"type":"function","title":"dbg.cn/1","doc":"Clears a node from the list of traced nodes.\n\n`cn` stands for **c**lear **n**ode.\n\nSubsequent calls to `tp/2` and `p/2` will not consider that node, but tracing\nalready activated on the node will continue to be in effect.\n\nReturns `ok`. This call cannot fail.","ref":"dbg.html#cn/1"},{"type":"function","title":"dbg.ctp/0","doc":"","ref":"dbg.html#ctp/0"},{"type":"function","title":"dbg.ctp/1","doc":"Disables call tracing for one or more functions specified by `ModuleOrMFA`.\n\nIf `ModuleOrMFA` is an atom (a module name), this function call is\nequivalent to `ctp({ModuleOrMFA, '_', '_'})`.\n\nOtherwise, `ModuleOrMFA` should be `{Module, Function, Arity}`.\n\n`ctp` stands for **c**lear **t**race **p**attern.\n\nThe semantics of `ModuleOrMFA` is the same as for the corresponding function\nspecification in `tp/2` or `tpl/2`. Both local and global call trace\nis disabled.\n\nThe return value reflects how many functions that matched, and is constructed as\ndescribed in `tp/2`, except that no `{saved, N}` tuple is returned.","ref":"dbg.html#ctp/1"},{"type":"function","title":"dbg.ctp/2","doc":"","ref":"dbg.html#ctp/2"},{"type":"function","title":"dbg.ctp/3","doc":"","ref":"dbg.html#ctp/3"},{"type":"function","title":"dbg.ctpe/1","doc":"Clears match specifications for the specified trace event (`send` or\n`'receive'`), reverting to the default of tracing all triggered events.\n\n`ctpe` stands for **c**lear **t**race **p**attern **e**vent.","ref":"dbg.html#ctpe/1"},{"type":"function","title":"dbg.ctpg/0","doc":"","ref":"dbg.html#ctpg/0"},{"type":"function","title":"dbg.ctpg/1","doc":"Disables global call tracing for one or more functions specified by `ModuleOrMFA`.\n\nIf `ModuleOrMFA` is an atom (a module name), this function call is\nequivalent to `ctpg({ModuleOrMFA, '_', '_'})`.\n\nOtherwise, `ModuleOrMFA` should be `{Module, Function, Arity}`.\n\n`ctpg` stands for **c**lear **t**race **p**attern **g**lobal.\n\nThis function works as `ctp/1`, but only disables tracing set up with\n`tp/2` (not with `tpl/2`).","ref":"dbg.html#ctpg/1"},{"type":"function","title":"dbg.ctpg/2","doc":"","ref":"dbg.html#ctpg/2"},{"type":"function","title":"dbg.ctpg/3","doc":"","ref":"dbg.html#ctpg/3"},{"type":"function","title":"dbg.ctpl/0","doc":"","ref":"dbg.html#ctpl/0"},{"type":"function","title":"dbg.ctpl/1","doc":"Disables local call tracing for one or more functions specified by `ModuleOrMFA`.\n\nIf `ModuleOrMFA` is an atom (a module name), this function call is\nequivalent to `ctpl({ModuleOrMFA, '_', '_'})`.\n\nOtherwise, `ModuleOrMFA` should be `{Module, Function, Arity}`.\n\n`ctpl` stands for **c**lear **t**race **p**attern **l**ocal.\n\nThis function works as `ctp/1`, but only disables tracing set up with\n`tpl/2` (not with `tp/2`).","ref":"dbg.html#ctpl/1"},{"type":"function","title":"dbg.ctpl/2","doc":"","ref":"dbg.html#ctpl/2"},{"type":"function","title":"dbg.ctpl/3","doc":"","ref":"dbg.html#ctpl/3"},{"type":"function","title":"dbg.dtp/0","doc":"Forgets all match specifications saved during calls to `tp/2`.\n\n`dtp` stands for **d**elete **t**race **p**atterns.\n\nRemoving all saved match specifications is useful before restoring\nother match specifications from a file with `rtp/1`. Use `dtp/1` to\ndelete specific saved match specifications.","ref":"dbg.html#dtp/0"},{"type":"function","title":"dbg.dtp/1","doc":"Forgets a specific match specification saved during calls to `tp/2`.\n\n`dtp` stands for **d**elete **t**race **p**attern.","ref":"dbg.html#dtp/1"},{"type":"function","title":"dbg.flush_trace_port/0","doc":"","ref":"dbg.html#flush_trace_port/0"},{"type":"function","title":"dbg.flush_trace_port/1","doc":"","ref":"dbg.html#flush_trace_port/1"},{"type":"function","title":"dbg.fun2ms/1","doc":"Pseudo function that by means of a parse transform translates the\n_literal_ fun typed as parameter in the function call to a [match\nspecification](`e:erts:match_spec.md`).\n\nThe meaning of \"literal\" is that the fun needs to textually be written\nas the argument of the function call; it cannot be held in a variable\nwhich in turn is passed to the function. Furthermore, the parse\ntransform module `m:ms_transform` must be enabled. The easiest way to\nenable it is by adding the following line to the source file:\n\n```\n-include_lib(\"stdlib/include/ms_transform.hrl\").\n```\n\nFailing to include `ms_transform.hrl` in the source will result in a runtime\nerror, not a compile-time error.\n\nThis function can also be invoked directly from the Erlang shell, as shown in\nthe examples that follow.\n\nThe head of the fun must be a single pattern that matches a list. That pattern\nwill be used to match the arguments for the call:\n\n_Examples_:\n\n```erlang\n1> dbg:fun2ms(fun([_,_]) -> true end).\n[{['_','_'],[],[true]}]\n2> dbg:fun2ms(fun(Args) when length(Args) > 6 -> true end).\n[{'$1',[{'>',{length,'$1'},6}],[true]}]\n```\n\nThe first match specification matches when a function having two\narguments is called. The second matches when a function with more than\n6 arguments is called.\n\n_Examples_:\n\n```erlang\n1> dbg:fun2ms(fun(42) -> true end).\nError: dbg:fun2ms requires fun with single variable or list parameter\n{error,transform_error}\n2> dbg:fun2ms(fun([< >]) -> true end).\nError: fun head contains bit syntax matching of variable 'H', which cannot be translated into match_spec\n{error,transform_error}\n```\n\nThe preceding two examples show what happens when a fun cannot be\ntranslated into a match specification. In the first example, the fun\nhead connot possibly match a list. In the second example, an attempt is made\nto take apart a binary using the bit syntax, which is currently not\nsupported in match specifications.\n\nHowever, note that literal binaries *can* be matched:\n\n```erlang\n1> dbg:fun2ms(fun([<<\"abc\">>]) -> true end).\n[{[<<\"abc\">>],[],[true]}]\n```\n\nMatch specifications support a large subset of the\n[guard expressions](`e:system:expressions.md#guard-expressions`) supported\nby Erlang, but not all. For example, updating a map is currently not supported:\n\n```erlang\n1> dbg:fun2ms(fun([M]) when map_size(M#{a => b}) > 2 -> true end).\nError: the language element map (in guard) cannot be translated into match_spec\n{error,transform_error}\n```\n\nHowever, creating a map in a guard is allowed:\n\n```erlang\n1> dbg:fun2ms(fun([M]) when map_size(#{a => b}) > 2 -> true end).\n[{['$1'],[{'>',{map_size,#{a => b}},2}],[true]}]\n```\n\nVariables from the environment can be imported, so this works:\n\n```erlang\n1> X = 3.\n3\n2> dbg:fun2ms(fun([M,N]) when N > X  -> return_trace() end).\n[{['$1','$2'],[{'>','$2',{const,3}}],[{return_trace}]}]\n```\nThe imported variables will be replaced by `const` expressions, which\nis consistent with the static scoping for Erlang funs.\n\nIn the body of the fun, only guard expressions and calls to the\n[special functions for tracing](`e:erts:match_spec.md#functions-allowed-only-for-tracing`)\nare allowed.\n\n_Examples_:\n\n```erlang\n1> dbg:fun2ms(fun([A]) when is_atom(A) -> return_trace() end).\n[{['$1'],[{is_atom,'$1'}],[{return_trace}]}]\n2> dbg:fun2ms(fun(_) -> erlang:garbage_collect() end).\nError: fun containing the remote function call 'erlang:garbage_collect/0' (called in body) cannot be translated into match_spec\n{error,transform_error}\n```\n\n> #### Warning {: .warning }\n>\n> If the parse transform is not applied to a module which calls `dbg:fun2ms/1`,\n> the call will fail in runtime with a `badarg` exception.\n\nMore information is available in the documentation for module `m:ms_transform`\nin STDLIB.","ref":"dbg.html#fun2ms/1"},{"type":"function","title":"dbg.get_tracer/0","doc":"","ref":"dbg.html#get_tracer/0"},{"type":"function","title":"dbg.get_tracer/1","doc":"Returns the process, port, or tracer module to which all trace messages are sent.","ref":"dbg.html#get_tracer/1"},{"type":"function","title":"dbg.h/0","doc":"Gives a list of items for brief online help.\n\n`h` stands for **h**elp.","ref":"dbg.html#h/0"},{"type":"function","title":"dbg.h/1","doc":"Gives a brief help text for functions in the `dbg` module.\n\n`h` stands for **h**elp.\n\nThe available items can be listed by calling `dbg:h/0`.","ref":"dbg.html#h/1"},{"type":"function","title":"dbg.i/0","doc":"Displays information about all traced processes and ports.\n\n`i` stands for **i**nformation.","ref":"dbg.html#i/0"},{"type":"function","title":"dbg.ln/0","doc":"Shows the list of traced nodes on the console.\n\n`ln` stands for **l**ist **n**odes.","ref":"dbg.html#ln/0"},{"type":"function","title":"dbg.ltp/0","doc":"Lists all match specifications previously used in the session.\n\n`ltp` stands for **l**ist **t**race **p**atterns.\n\nThis function lists all match specifications previously saved during\ncalls to `tp/2` and `tpl/2`, as well as all built-in match\nspecifications. This avoids having to re-type complicated match\nspecifications. Note that the match specifications are lost if\n`stop/0` is called.\n\nMatch specifications can be saved in a file (if a read-write file system is\npresent) for use in later debugging sessions; see `wtp/1` and `rtp/1`.\n\nThere are three built-in trace patterns:\n\n- `exception_trace`, `x` - sets a trace which will show function\n  names, parameters, return values, and exceptions raised from\n  functions\n\n- `caller_trace`, `c` - sets a trace that displays function names,\n  parameters, and information about which function called it\n\n- `caller_exception_trace`, `cx` - combines `exception_trace` and\n  `caller_trace`\n\nHere is an example that shows how to use a built-in match specification:\n\n```erlang\n1> dbg:tracer().\n{ok,<0.90.0>}\n2> dbg:tp(lists, seq, 2, cx).\n{ok,[{matched,nonode@nohost,1},{saved,cx}]}\n3> dbg:p(self(), call).\n{ok,[{matched,nonode@nohost,1}]}\n4> lists:seq(1, 5).\n(<0.88.0>) call lists:seq(1,5) ({erl_eval,do_apply,7,{\"erl_eval.erl\",904}})\n[1,2,3,4,5]\n(<0.88.0>) returned from lists:seq/2 -> [1,2,3,4,5]\n```","ref":"dbg.html#ltp/0"},{"type":"function","title":"dbg.n/1","doc":"Adds a remote node (`Nodename`) to the list of nodes where tracing is\nperformed.\n\n`n` stands for **n**ode.\n\nThe `dbg` server keeps a list of nodes where tracing should be\nperformed. Whenever a `tp/2` call or a `p/2` call is made, it is\nexecuted for all nodes in this list including the local node (except\nfor `p/2` with a specific `t:pid/0` or `t:port/0` as first argument,\nin which case the command is executed only on the node where the\ndesignated process or port resides).\n\nWhen this function is called, it starts a tracer process on the remote\nnode, which will send all trace messages to the tracer process on the\nlocal node (via the Erlang distribution). If no tracer process is\nrunning on the local node, the error reason `no_local_tracer` is\nreturned. The tracer process on the local node must be started with\nthe [`tracer/0,2`](`tracer/2`) function.\n\nIf `Nodename` is the local node, the error reason `cant_add_local_node` is\nreturned.\n\nIf a trace port (see `trace_port/2`) is running on the local node, remote nodes\ncannot be traced with a tracer process. The error reason\n`cant_trace_remote_pid_to_local_port` is returned. However, a trace port can be\nstarted on the remote node with the `tracer/3` function.\n\nThe function will also return an error if the node `Nodename` is not reachable.","ref":"dbg.html#n/1"},{"type":"function","title":"dbg.p/1","doc":"","ref":"dbg.html#p/1"},{"type":"function","title":"dbg.p/2","doc":"Traces `Item` in accordance to the value specified by `Flags`.\n\n`p` stands for **p**rocess.\n\nThe following kind of values are allowed for `Item`:\n\n- **`t:pid/0` or `t:port/0`** - The corresponding process or port is traced. The\n  process or port can be a remote process or port (on another Erlang node). The\n  node must be in the list of traced nodes (see `n/1` and `tracer/3`).\n\n- **`all`** - All processes and ports in the system as well as all processes and\n  ports created hereafter are to be traced.\n\n- **`processes`** - All processes in the system as well as all processes created\n  hereafter are to be traced.\n\n- **`ports`** - All ports in the system as well as all ports created hereafter\n  are to be traced.\n\n- **`new`** - All processes and ports created after the call are to be\n  traced.\n\n- **`new_processes`** - All processes created after the call are to be\n  traced.\n\n- **`new_ports`** - All ports created after the call are to be traced.\n\n- **`existing`** - All existing processes and ports are traced.\n\n- **`existing_processes`** - All existing processes are traced.\n\n- **`existing_ports`** - All existing ports are traced.\n\n- **`t:atom/0`** - The process or port with the corresponding registered name is\n  traced. The process or port can on another Erlang node.\n  The node must be in the list of traced nodes (see `n/1` and `tracer/3`).\n\n- **`t:integer/0`** - The process `<0.Item.0>` is traced.\n\n- **`{X, Y, Z}`** - The process ` ` is traced.\n\n- **`t:string/0`** - If the `Item` is a string \" \" as returned from\n  [`pid_to_list/1`](`erlang:pid_to_list/1`), the process ` ` is traced.\n\nWhen enabling an `Item` that represents a group of processes, the `Item` is\nenabled on all nodes added with the `n/1` or `tracer/3` function.\n\n`Flags` can be a single atom or a list of flags. The available flags are:\n\n- **`s (send)`** - Traces the messages the process or port sends.\n\n- **`r (receive)`** - Traces the messages the process or port receives.\n\n- **`m (messages)`** - Traces the messages the process or port receives and\n  sends.\n\n- **`c (call)`** - Traces global function calls for the process according to the\n  trace patterns set in the system (see `tp/2`).\n\n- **`p (procs)`** - Traces process related events to the process.\n\n- **`ports`** - Traces port related events to the port.\n\n- **`sos (set on spawn)`** - Lets all processes created by the traced process\n  inherit the trace flags of the traced process.\n\n- **`sol (set on link)`** - Lets another process, `P2`, inherit the trace flags\n  of the traced process whenever the traced process links to `P2`.\n\n- **`sofs (set on first spawn)`** - This is the same as `sos`, but only for the\n  first process spawned by the traced process.\n\n- **`sofl (set on first link)`** - This is the same as `sol`, but only for the\n  first call to [`link/1`](`erlang:link/1`) by the traced process.\n\n- **`all`** - Sets all flags except `silent`.\n\n- **`clear`** - Clears all flags.\n\nThe list can also include any of the flags allowed in `trace:process/4` and\n`trace:port/4`.\n\nThis function returns either an error tuple or an `{ok, List}` tuple. The `List`\nconsists of specifications of how many processes and ports that matched (in the\ncase of a single pid exactly 1). The specification of matched processes is\n`{matched, Node, N}`. If the remote processor call (using `m:rpc`) to a remote node\nfails, the `rpc` error message is returned as the fourth element in the tuple\nand the number of matched processes is 0.","ref":"dbg.html#p/2"},{"type":"function","title":"dbg.rtp/1","doc":"Reads match specifications from a text file (possibly) generated by\nthe `wtp/1` function.\n\n`rtp` stands for **r**ead **t**race **p**atterns.\n\nThe function verifies that the syntax of all match specifications are correct.\nIf any error in any match specification is found, none of the match specifications\nare added to the list of saved match specifications for the running system.\n\nThe match specifications in the file are _merged_ with the current match\nspecifications, so that no duplicates are generated. Use `ltp/0` to see what\nnumbers were assigned to the specifications from the file.\n\nThe function will return an error tuple, either due to I/O problems\n(like a non-existing or non-readable file) or due to file format\nproblems. In the latter case, `Reason` is in a more or less textual\nformat, giving a hint to what is causing the problem.","ref":"dbg.html#rtp/1"},{"type":"function","title":"dbg.session/2","doc":"Runs `m:dbg` commands using the provides session, or\ncreates a session for the duration of the call if a session name\nis provided.\n\nAny `m:dbg` function that is called with in the provided fun\nwill use the `t:session/0` provided instead of the default\n`dbg` session. This means that the tracing will be isolated\nfrom other tracing users on the system.\n\nThe function returns the term that the fun returns.\n\n*Example*:\n\n```erlang\n1> S = dbg:session_create(my_session).\n<0.91.0>\n2> dbg:session(S, fun() -> dbg:tracer(), dbg:p(all,c), dbg:tp(lists,seq,x) end).\n{ok,[{matched,nonode@nohost,2},{saved,x}]}\n3> lists:seq(1, 10).\n(<0.89.0>) call lists:seq(1,10)\n(<0.89.0>) returned from lists:seq/2 -> [1,2,3,4,5,6,7,8,9,10]\n[1,2,3,4,5,6,7,8,9,10]\n4> dbg:session_destroy(S).\nok\n```\n\nThe state of the `t:session/0` is preserved in between `session/2` calls, so\nyou can call `session/2` multiple when debugging you application.\n\n*Example*:\n\n```erlang\n1> S = dbg:session_create(my_session).\n<0.91.0>\n%% Setup the initial traces\n2> dbg:session(S, fun() -> dbg:tracer(), dbg:p(self(),c), dbg:tp(lists,seq,x) end).\n{ok,[{matched,nonode@nohost,2},{saved,x}]}\n3> lists:seq(1, 3).\n(<0.89.0>) call lists:seq(1,3)\n(<0.89.0>) returned from lists:seq/2 -> [1,2,3]\n[1,2,3]\n%% Add an additional trace pattern\n4> dbg:session(S, fun() -> dbg:tpl(lists,seq_loop,x) end).\nok\n5> lists:seq(1, 3).\n(<0.89.0>) call lists:seq(1,3)\n(<0.89.0>) call lists:seq_loop(3,3,[])\n(<0.89.0>) call lists:seq_loop(1,1,[2,3])\n(<0.89.0>) returned from lists:seq_loop/3 -> [1,2,3]\n(<0.89.0>) returned from lists:seq_loop/3 -> [1,2,3]\n(<0.89.0>) returned from lists:seq/2 -> [1,2,3]\n[1,2,3]\n6> dbg:session_destroy(S).\nok\n```\n\n> #### Note {: .info }\n>\n> The session functionality is experimental in Erlang/OTP 27\n> and may change in future releases without notice.","ref":"dbg.html#session/2"},{"type":"function","title":"dbg.session_create/1","doc":"Create a new `m:dbg` session with the given `Name`.\n\nThe session is linked with the calling process and will be\n\nMultiple sessions can have the same name.\n\n> #### Note {: .info }\n>\n> The session functionality is experimental in Erlang/OTP 27\n> and may change in future releases without notice.","ref":"dbg.html#session_create/1"},{"type":"function","title":"dbg.session_destroy/1","doc":"Destroys a dbg `t:session/0`.\n\nThis will terminate all started processes and destroy the `t:trace:session/0`.","ref":"dbg.html#session_destroy/1"},{"type":"function","title":"dbg.stop/0","doc":"Stops the `dbg` server, clears all trace flags for all processes, clears all\ntrace patterns for all functions, clears trace patterns for send/receive, shuts\ndown all trace clients, and closes all trace ports.","ref":"dbg.html#stop/0"},{"type":"function","title":"dbg.stop_trace_client/1","doc":"Shuts down a previously started trace client.\n\nThe `Pid` argument is the process id returned from the\n`trace_client/2` or `trace_client/3` call.","ref":"dbg.html#stop_trace_client/1"},{"type":"function","title":"dbg.tp/2","doc":"Enables call trace for one or more exported functions specified by `ModuleOrMFA`.\n\nIf `ModuleOrMFA` is an atom (a module name), this function call is equivalent to\n`tp({ModuleOrMFA, '_', '_'}, MatchSpec)`.\n\nOtherwise, `ModuleOrMFA` should be `{Module, Function, Arity}`.\n\n`tp` stands for **t**race **p**attern.\n\nAll exported functions matching the `{Module, Function, Arity}`\nargument will be concerned, but the match specification may further\nnarrow down the set of function calls generating trace messages.\n\nFor a description of the format for the `MatchSpec` argument, see\n[_Match Specifications in Erlang_](`e:erts:match_spec.md`), which explains the\ngeneral match specification language. The most common generic match\nspecifications used can be found as built-in aliases; see `ltp/0` below for\ndetails.\n\nThe Module, Function and/or Arity parts of the tuple may be specified\nas the atom `'_'` which is a wildcard matching all modules, functions,\nor arities. Note that if the `Module` is specified as `'_'`, the\n`Function` and `Arity` parts must be specified as `'_'` as well. The\nsame holds for the `Function` in relation to `Arity`.\n\nAll nodes added with `n/1` or `tracer/3` will be affected by this call, and if\n`Module` is not `'_'` the module will be loaded on all nodes.\n\nThe function returns either an error tuple or an `{ok, List}` tuple. The `List`\nconsists of specifications of how many functions that matched, in the same way\nas the processes and ports are presented in the return value of `p/2`.\n\nThere may be a tuple `{saved, N}` in the return value, if the `MatchSpec` is not\n`[]`. The integer `N` can then be used in subsequent calls to this function\nand will stand as an \"alias\" for the given expression.\n\nIf the match specification is invalid, an `{error, Errors}` tuple is\nreturned.  `Errors` is as a list of tuples `{error, string()}`, where\nthe string is a textual explanation of the compilation error. For\nexample:\n\n```erlang\n1> dbg:tp({dbg,ltp,0},[{[],[],[{message, two, arguments}, {noexist}]}]).\n{error,\n [{error,\"Special form 'message' called with wrong number of\n          arguments in {message,two,arguments}.\"},\n  {error,\"Function noexist/1 does_not_exist.\"}]}\n```","ref":"dbg.html#tp/2"},{"type":"function","title":"dbg.tp/3","doc":"","ref":"dbg.html#tp/3"},{"type":"function","title":"dbg.tp/4","doc":"","ref":"dbg.html#tp/4"},{"type":"function","title":"dbg.tpe/2","doc":"Associates a match specification with trace event `send` or\n`'receive'`.\n\n`tpe` stands for **t**race **p**attern **e**vent.\n\nBy default all executed\n`send` and `'receive'` events are traced if enabled for a process. A match\nspecification can be used to filter traced events based on sender, receiver,\nand/or message content.\n\nFor a description of the format for the `MatchSpec` argument, see\n[_Match Specifications in Erlang_](`e:erts:match_spec.md`), which explains the\ngeneral match specification language.\n\nFor `send`, the matching is done on the list `[Receiver, Msg]`. `Receiver` is\nthe process or port identity of the receiver and `Msg` is the message term. The\npid of the sending process can be accessed with the guard function `self/0`.\n\nFor `'receive'`, the matching is done on the list `[Node, Sender, Msg]`. `Node`\nis the node name of the sender. `Sender` is the process or port identity of the\nsender, or the atom `undefined` if the sender is not known (which may be the\ncase for remote senders). `Msg` is the message term. The pid of the receiving\nprocess can be accessed by calling `self/0`.\n\nAll nodes added with `n/1` or `tracer/3` will be affected by this call.\n\nThe return value is the same as for `tp/2`. The number of matched events is\nalways 1 as [`tpe/2`](`tpe/2`) does not accept any form of wildcards\nfor argument `Event`.","ref":"dbg.html#tpe/2"},{"type":"function","title":"dbg.tpl/2","doc":"Enables call trace for one or more functions specified by `ModuleOrMFA`.\n\nIf `ModuleOrMFA` is an atom (a module name), this function call is equivalent to\n`tpl({ModuleOrMFA, '_', '_'}, MatchSpec)`.\n\nOtherwise, `ModuleOrMFA` should be `{Module, Function, Arity}`.\n\n`tpl` stands for **t**race **p**attern **l**ocal.\n\nThis function works as `tp/2`, but enables tracing for local or remote calls\nto both local and exported functions.","ref":"dbg.html#tpl/2"},{"type":"function","title":"dbg.tpl/3","doc":"","ref":"dbg.html#tpl/3"},{"type":"function","title":"dbg.tpl/4","doc":"","ref":"dbg.html#tpl/4"},{"type":"function","title":"dbg.trace_client/2","doc":"Starts a trace client that reads the output created by a trace port\ndriver (see `trace_port/2`) and handles it in mostly the same way as a\ntracer process created by the `tracer/0` function.\n\nIf `Type` is `file`, the client reads all trace messages stored in the\nfile named `Filename` or specified by `WrapFilesSpec` (must be the\nsame as used when creating the trace) and lets the default handler\nfunction format the messages on the console. This is one way to\ninterpret the data stored in a file by the file trace port driver.\n\nIf `Type` is `follow_file`, the client behaves as in the `file` case, but keeps\ntrying to read (and process) more data from the file until stopped by\n`stop_trace_client/1`. `WrapFilesSpec` is not allowed as second argument for\nthis `Type`.\n\nIf `Type` is `ip`, the client connects to the TCP/IP port `PortNumber` on the\nhost `Hostname`, from where it reads trace messages until the TCP/IP connection\nis closed. If no `Hostname` is specified, the local host is assumed.\n\nAs an example, one can let trace messages be sent over the network to another\nErlang node (preferably _not_ distributed), where the formatting occurs.\n\nOn the node `stack` there exists an Erlang node `ant@stack`. In the\nshell, type the following:\n\n```erlang\nant@stack> dbg:tracer(port, dbg:trace_port(ip, 4711)).\n<0.17.0>\nant@stack> dbg:p(self(), send).\n{ok,1}\n```\n\nAll trace messages are now sent to the trace port driver, which in turn listens\nfor connections on the TCP/IP port 4711. If we want to see the messages on\nanother node, preferably on another host, we do like this:\n\n```erlang\n1> dbg:trace_client(ip, {\"stack\", 4711}).\n<0.42.0>\n```\n\nIf we now send a message from the shell on the node `ant@stack`, where all sends\nfrom the shell are traced:\n\n```erlang\nant@stack> self() ! hello.\nhello\n```\n\nThe following will appear at the console on the node that started the trace\nclient:\n\n```erlang\n(<0.23.0>) <0.23.0> ! hello\n(<0.23.0>) <0.22.0> ! {shell_rep,<0.23.0>,{value,hello,[],[]}}\n```\n\nThe last line is generated due to internal message passing in the Erlang shell.\nThe pids will vary.","ref":"dbg.html#trace_client/2"},{"type":"function","title":"dbg.trace_client/3","doc":"This function works exactly as `trace_client/2`, but allows you to write your\nown handler function.\n\nThe handler function works mostly as the one described in `tracer/2`,\nbut must also be prepared to handle trace messages of the form `{drop,\nN}`, where `N` is the number of dropped messages. This pseudo trace\nmessage will only occur if the `ip` trace driver is used.\n\nFor trace type `file`, the pseudo trace message `end_of_trace` will appear at\nthe end of the trace. The return value from the handler function is in this case\nignored.","ref":"dbg.html#trace_client/3"},{"type":"function","title":"dbg.trace_port/2","doc":"Creates a trace-port-generating fun that is suitable as the\nsecond argument to `tracer/2`.\n\n_Example:_\n\n```erlang\ndbg:tracer(port, dbg:trace_port(ip, 4711)).\n```\n\nA trace port is an Erlang port to a dynamically linked-in driver that\nhandles trace messages directly, without the overhead of sending them\nas messages to an Erlang process. Using a trace port significantly\nlowers the overhead imposed by tracing.\n\nTwo trace drivers are currently implemented: the `file` and the `ip`\ntrace drivers.\n\nThe `file` driver sends all trace messages into one or\nseveral binary files, from where they later can be fetched and\nprocessed with the `trace_client/2` function.\n\nThe `ip` driver opens a TCP/IP port listening port. When a client\n(preferably started by calling `trace_client/2` on another Erlang\nnode) connects, all trace messages are sent over the IP network for\nfurther processing by the remote client.\n\nThe `file` trace driver expects a filename or a wrap files\nspecification as parameter. A file is written with a high degree of\nbuffering, which is why there is no guarantee that all are saved in the\nfile in case of a system crash.\n\nA wrap files specification is used to limit the disk space consumed by the\ntrace. The trace is written to a limited number of files each with a limited\nsize. The actual filenames are `Filename ++ SeqCnt ++ Suffix`, where `SeqCnt`\ncounts as a decimal string from `0` to `WrapCnt` and then around again from `0`.\nWhen a trace term written to the current file makes it longer than `WrapSize`,\nthat file is closed, and if the number of files in this wrap trace is as many as\n`WrapCnt` the oldest file is deleted, and a new file is opened to become the\ncurrent. Thus, when a wrap trace has been stopped, there are at most `WrapCnt`\ntrace files saved with a size of at least `WrapSize` (but not much larger),\nexcept for the last file that might even be empty. The default values are\n`WrapSize = 128*1024` and `WrapCnt = 8`.\n\nThe `SeqCnt` values in the filenames are all in the range `0` through `WrapCnt`\nwith a gap in the circular sequence. The gap is needed to find the end of the\ntrace.\n\nIf the `WrapSize` is specified as `{time, WrapTime}`, the current file is closed\nwhen it has been open more than `WrapTime` milliseconds, regardless of it being\nempty or not.\n\nThe `ip` trace driver has a queue of `QueSize` messages waiting to be delivered.\nIf the driver cannot deliver messages as fast as they are produced by the\nruntime system, a special message is sent, which indicates how many messages\nthat are dropped. That message will arrive at the handler function specified in\n`trace_client/3` as the tuple `{drop, N}` where `N` is the number of consecutive\nmessages dropped. In case of heavy tracing, drops are likely to occur, and they\nsurely occur if no client is reading the trace messages. The default value of\n`QueSize` is 200.","ref":"dbg.html#trace_port/2"},{"type":"function","title":"dbg.trace_port_control/1","doc":"","ref":"dbg.html#trace_port_control/1"},{"type":"function","title":"dbg.trace_port_control/2","doc":"This function is used to do a control operation on the active trace port driver\non the given node (`Nodename`).\n\nWhich operations are allowed as well as their return values depend on\nwhich trace driver is used.\n\nReturns either `ok` or `{ok, Result}` if the operation was successful, or\n`{error, Reason}` if the current tracer is a process, or if it is a port not\nsupporting the operation.\n\nThe allowed values for `Operation` are:\n\n- **`flush`** - This function is used to flush the internal buffers held by a\n  trace port driver. Currently only the `file` trace driver supports this\n  operation. Returns `ok`.\n\n- **`get_listen_port`** - Returns `{ok, IpPort}` where `IpPort` is the IP port\n  number used by the driver listen socket. Only the `ip` trace driver supports\n  this operation.","ref":"dbg.html#trace_port_control/2"},{"type":"function","title":"dbg.tracer/0","doc":"Starts a server on the local node that will be the recipient of\nall trace messages.\n\nAll subsequent calls to `p/2` will result in messages sent\nto the newly started trace server.\n\nA trace server started in this way will simply display the formatted\ntrace messages the Erlang shell (that is, using `io:format/2`). See `tracer/2`\nfor a description of how the trace message handler can be customized.\n\nTo start a similar tracer on a remote node, use `n/1`.","ref":"dbg.html#tracer/0"},{"type":"function","title":"dbg.tracer/2","doc":"Starts a tracer server with additional parameters on the local\nnode.\n\n`Type` indicates how trace messages should be handled:\n\n- `process` - by a receiving process\n- `port` - by a port; see `trace_port/2`\n- `module` - by a tracer module; see `m:erl_tracer`\n- `file` - by printing them to a file\n\nIf `Type` is `process`, `Data` should be a message handler function\n(`HandlerSpec`). The handler function, which should be a `fun` taking two\narguments, will be called for each trace message, with the first argument\ncontaining the message as it is and the second argument containing the return\nvalue from the last invocation of the fun. The initial value of the second\nparameter is specified in the `InitialData` part of the `HandlerSpec`. The\n`HandlerFun` can choose any appropriate action to take when invoked, and can\nsave a state for the next invocation by returning it.\n\nIf `Type` is `port`, then the second parameter should be a fun which takes no\narguments and returns a newly opened trace port when called. Such a fun is\npreferably generated by calling `trace_port/2`.\n\nIf `Type` is `module`, `Data` should be either a tuple describing the\n`m:erl_tracer` module to be used for tracing and the state to be used\nfor that tracer module, or a fun returning that kind of tuple.\n\nif `Type` is `file`, `Data` should be a filename specifying a file\nwhere all the traces are to be printed.\n\nIf an error is returned, it can either be because a tracer server is\nalready running (`{error,already_started}`), or because\n`HandlerFun` raised an exception.\n\nTo start a similar tracer on a remote node, use `tracer/3`.","ref":"dbg.html#tracer/2"},{"type":"function","title":"dbg.tracer/3","doc":"This function is equivalent to `tracer/2`, but acts on the given node.\n\nA tracer is started on the node (`Nodename`) and the node is added to\nthe list of traced nodes.\n\n> #### Note {: .info }\n>\n> This function is not equivalent to `n/1`. While `n/1` starts a process tracer\n> which redirects all trace information to a process tracer on the local node\n> (that is, the trace control node), `tracer/3` starts any type of tracer,\n> independent of the type of tracer on the trace control node.\n\nFor details, see `tracer/2`.","ref":"dbg.html#tracer/3"},{"type":"function","title":"dbg.wtp/1","doc":"Saves all match specifications saved during the session (by calls to\n`tp/2` or `tpl/2`), as well as built-in match specifications, in a text\nfile with the name designated by `Name`.\n\n`wtp` stands for **w**rite **t**race **p**atterns.\n\nThe format of the file is textual, which means that it can be edited\nwith a text editor, and then restored with `rtp/1`.\n\nEach match specification in the file ends with a period (`.`) and\nnew (syntactically correct) match specifications can be added to the\nfile manually.\n\nThe function returns `ok`, or an error tuple where the second element\nindicates the reason that writing the file failed.","ref":"dbg.html#wtp/1"},{"type":"type","title":"dbg.built_in_alias/0","doc":"","ref":"dbg.html#t:built_in_alias/0"},{"type":"type","title":"dbg.handler_spec/0","doc":"","ref":"dbg.html#t:handler_spec/0"},{"type":"type","title":"dbg.match_desc/0","doc":"","ref":"dbg.html#t:match_desc/0"},{"type":"type","title":"dbg.match_info/0","doc":"","ref":"dbg.html#t:match_info/0"},{"type":"type","title":"dbg.match_num/0","doc":"","ref":"dbg.html#t:match_num/0"},{"type":"type","title":"dbg.match_pattern/0","doc":"","ref":"dbg.html#t:match_pattern/0"},{"type":"type","title":"dbg.match_spec/0","doc":"","ref":"dbg.html#t:match_spec/0"},{"type":"opaque","title":"dbg.session/0","doc":"A `m:dbg` session that can be used by `session/2` to\ncreate isolated debugging sessions.","ref":"dbg.html#t:session/0"},{"type":"type","title":"dbg.tp_arity/0","doc":"","ref":"dbg.html#t:tp_arity/0"},{"type":"type","title":"dbg.tp_function/0","doc":"","ref":"dbg.html#t:tp_function/0"},{"type":"type","title":"dbg.tp_id/0","doc":"","ref":"dbg.html#t:tp_id/0"},{"type":"type","title":"dbg.tp_match_spec/0","doc":"","ref":"dbg.html#t:tp_match_spec/0"},{"type":"type","title":"dbg.tp_module/0","doc":"","ref":"dbg.html#t:tp_module/0"},{"type":"type","title":"dbg.trace_wrap_file_size/0","doc":"","ref":"dbg.html#t:trace_wrap_file_size/0"},{"type":"type","title":"dbg.trace_wrap_files_spec/0","doc":"","ref":"dbg.html#t:trace_wrap_files_spec/0"},{"type":"module","title":"dyntrace","doc":"Interface to dynamic tracing\n\nThis module implements interfaces to dynamic tracing, should such be compiled\ninto the virtual machine. For a standard and/or commercial build, no dynamic\ntracing is available, in which case none of the functions in this module is\nusable or give any effect.\n\nShould dynamic tracing be enabled in the current build, either by configuring\nwith `./configure --with-dynamic-trace=dtrace` or with\n`./configure --with-dynamic-trace=systemtap`, the module can be used for two\nthings:\n\n- Trigger the user-probe `user_trace_i4s4` in the NIF library `dyntrace.so` by\n  calling `dyntrace:p/{1,2,3,4,5,6,7,8}`.\n- Set a user specified tag that will be present in the trace messages of both\n  the `efile_drv` and the user-probe mentioned above.\n\nBoth building with dynamic trace probes and using them is experimental and\nunsupported by Erlang/OTP. It is included as an option for the developer to\ntrace and debug performance issues in their systems.\n\nThe original implementation is mostly done by Scott Lystiger Fritchie as an Open\nSource Contribution and it should be viewed as such even though the source for\ndynamic tracing as well as this module is included in the main distribution.\nHowever, the ability to use dynamic tracing of the virtual machine is a very\nvaluable contribution which OTP has every intention to maintain as a tool for\nthe developer.\n\nHow to write `d` programs or `systemtap` scripts can be learned from books and\nfrom a lot of pages on the Internet. This manual page does not include any\ndocumentation about using the dynamic trace tools of respective platform.\nHowever, the `examples` directory of the `runtime_tools` application contains\ncomprehensive examples of both `d` and `systemtap` programs that will help you\nget started. Another source of information is the [dtrace](dtrace.md) and\n[systemtap](systemtap.md) chapters in the Runtime Tools Users' Guide.","ref":"dyntrace.html"},{"type":"function","title":"dyntrace.available/0","doc":"This function uses the NIF library to determine if dynamic tracing is available.\n\nThis function will throw an exception if the `dyntrace` NIF library could not be\nloaded by the `on_load` function in this module.\n\nUse [`erlang:system_info(dynamic_trace)`](`e:erts:erlang.md#system_info_dynamic_trace`)\nto determine whether the run-time system supports dynamic tracing.","ref":"dyntrace.html#available/0"},{"type":"function","title":"dyntrace.get_tag/0","doc":"This function returns the user tag set in the current process. If no tag is set\nor dynamic tracing is not available, it returns `undefined`.\n\nThis function returns the user tag set in the current process or, if no user tag\nis present, the last user tag sent to the process together with a message (in\nthe same way as [sequential trace tokens](`m:seq_trace`) are spread to other\nprocesses together with messages. For an explanation of how user tags can be\nspread together with messages, see `spread_tag/1`. If no tag is found or dynamic\ntracing is not available, it returns `undefined`","ref":"dyntrace.html#get_tag/0"},{"type":"function","title":"dyntrace.p/0","doc":"Calling this function triggers the \"user\" trace probe `user_trace_i4s4` in the\ndyntrace NIF module, sending a trace message containing only the user tag and\nzeroes/empty strings in all other fields.","ref":"dyntrace.html#p/0"},{"type":"function","title":"dyntrace.p/1","doc":"Calling this function will trigger the \"user\" trace probe `user_trace_i4s4` in the\n`dyntrace` NIF module, sending a trace message containing the user tag and the\ninteger or string parameter in the first integer/string field.","ref":"dyntrace.html#p/1"},{"type":"function","title":"dyntrace.p/2","doc":"Calling this function will trigger the \"user\" trace probe `user_trace_i4s4` in the\ndyntrace NIF module, sending a trace message containing the user tag and the\n[`integer()`](`t:integer/0`) or [`string()`](`t:string/0`) parameters as the\nfirst fields of their respective type.\n\n[`integer()`](`t:integer/0`) parameters should be put before any\n[`string()`](`t:string/0`) parameters.\n\nThat is, the following calls work:\n\n- [`dyntrace:p(1, \"Hello\")`](`p/2`)\n- [`dyntrace:p(1, 1)`](`p/2`)\n- [`dyntrace:p(\"Hello\", \"Again\")`](`p/2`)\n\nThe following call is invalid because the string argument comes before the\ninteger argument:\n\n- [`dyntrace:p(\"Hello\", 1)`](`p/2`)","ref":"dyntrace.html#p/2"},{"type":"function","title":"dyntrace.p/3","doc":"Calling this function will trigger the \"user\" trace probe `user_trace_i4s4` in the\ndyntrace NIF module, sending a trace message containing the user tag and the\n[`integer()`](`t:integer/0`) or [`string()`](`t:string/0`) parameters as the\nfirst fields of their respective type.\n\n[`integer()`](`t:integer/0`) parameters should be put before any\n[`string()`](`t:string/0`) parameters.","ref":"dyntrace.html#p/3"},{"type":"function","title":"dyntrace.p/4","doc":"Calling this function will trigger the \"user\" trace probe `user_trace_i4s4` in the\ndyntrace NIF module, sending a trace message containing the user tag and the\n[`integer()`](`t:integer/0`) or [`string()`](`t:string/0`) parameters as the\nfirst fields of their respective type.\n\n[`integer()`](`t:integer/0`) parameters should be put before any\n[`string()`](`t:string/0`) parameters.","ref":"dyntrace.html#p/4"},{"type":"function","title":"dyntrace.p/5","doc":"Calling this function will trigger the \"user\" trace probe `user_trace_i4s4` in the\ndyntrace NIF module, sending a trace message containing the user tag and the\n[`integer()`](`t:integer/0`) or [`string()`](`t:string/0`) parameters as the\nfirst fields of their respective type.\n\n[`integer()`](`t:integer/0`) parameters should be put before any\n[`string()`](`t:string/0`) parameters.\n\nThere can be no more than four parameters of each type,\nso the first parameter must be of type [`integer()`](`t:integer/0`) and\nthe last parameter of type [`string()`](`t:string/0`).","ref":"dyntrace.html#p/5"},{"type":"function","title":"dyntrace.p/6","doc":"Calling this function will trigger the \"user\" trace probe `user_trace_i4s4` in the\ndyntrace NIF module, sending a trace message containing the user tag and the\n[`integer()`](`t:integer/0`) or [`string()`](`t:string/0`) parameters as the\nfirst fields of their respective type.\n\n[`integer()`](`t:integer/0`) parameters should be put before any\n[`string()`](`t:string/0`) parameters.\n\nThere can be no more than four parameters of each type,\nso the first two parameters must be of type [`integer()`](`t:integer/0`) and\nthe last two of type [`string()`](`t:string/0`).","ref":"dyntrace.html#p/6"},{"type":"function","title":"dyntrace.p/7","doc":"Calling this function will trigger the \"user\" trace probe `user_trace_i4s4` in the\ndyntrace NIF module, sending a trace message containing the user tag and the\n[`integer()`](`t:integer/0`) or [`string()`](`t:string/0`) parameters as the\nfirst fields of their respective type.\n\n[`integer()`](`t:integer/0`) parameters should be put before any\n[`string()`](`t:string/0`) parameters.\n\nThere can be no more than four parameters of each type,\nso the first three parameters must be of type [`integer()`](`t:integer/0`) and\nthe last three of type [`string()`](`t:string/0`).","ref":"dyntrace.html#p/7"},{"type":"function","title":"dyntrace.p/8","doc":"Calling this function will trigger the \"user\" trace `probe user_trace_i4s4` in the\ndyntrace NIF module, sending a trace message containing all the\n[`integer()`](`t:integer/0`) and [`string()`](`t:string/0`) parameters\nprovided, as well as any user tag set in the current process.","ref":"dyntrace.html#p/8"},{"type":"function","title":"dyntrace.put_tag/1","doc":"This function sets the user tag of the current process.\n\nThe user tag is a [`binary()`](`t:binary/0`), but can be specified as\nany [`iodata()`](`t:iodata/0`), which is automatically converted to a\nbinary by this function.\n\nThe user tag is provided to the user probes triggered by calls top\n`dyntrace:p/{1,2,3,4,5,6,7,8}` as well as probes in the `efile` driver. In the\nfuture, user tags might be added to more probes.\n\nThe old user tag (if any) is returned, or `undefined` if no user tag was present,\nor dynamic tracing is not enabled.","ref":"dyntrace.html#put_tag/1"},{"type":"function","title":"dyntrace.restore_tag/1","doc":"Restores the previous state of user tags and their spreading as it was before a\ncall to `spread_tag/1`.\n\nNote that the restoring is not limited to the same process; one can\nutilize this to turn off spreding in one process and restore it in a\nnewly created process that is is actually going to send messages:\n\n```erlang\nf() ->\n    TagData = dyntrace:spread_tag(false),\n    spawn(fun() ->\n             dyntrace:restore_tag(TagData),\n             do_something()\n          end),\n    do_something_else(),\n    dyntrace:restore_tag(TagData).\n```\n\nCorrectly handling user tags and their spreading might take some effort, as\nErlang programs tend to send and receive messages so that sometimes the user tag\ngets lost due to various things, like double receives or communication with a\nport (ports do not handle user tags, in the same way as they do not handle\nregular sequential trace tokens).","ref":"dyntrace.html#restore_tag/1"},{"type":"function","title":"dyntrace.spread_tag/1","doc":"This function controls if user tags are to be spread to other processes with the\nnext message.\n\nSpreading of user tags work like spreading of sequential trace\ntokens, so that a received user tag will be active in the process until the next\nmessage arrives (if that message does not also contain the user tag).\n\nThis functionality is used when a client process communicates with a file\ni/o-server to spread the user tag to the I/O-server and then down to the\n`efile` driver. By using [`spread_tag/1`](`spread_tag/1`) and\n[`restore_tag/1`](`restore_tag/1`), one can enable or disable spreading of user\ntags to other processes and then restore the previous state of the user tag. The\nTagData returned from this call contains all previous information so the state\n(including any previously spread user tags) will be completely restored by a\nlater call to [`restore_tag/1`](`restore_tag/1`).\n\nThe `m:file` module already spreads tags, so there is no need to manually call\nthis function to get user tags spread to the `efile` driver through that module.\n\nThe most use of this function would be if one, for example, uses the `m:io` module\nto communicate with an I/O-server for a regular file, such as in the following\nexample:\n\n```erlang\nf() ->\n   {ok, F} = file:open(\"test.tst\", [write]),\n   Saved = dyntrace:spread_tag(true),\n   io:format(F, \"Hello world!\", []),\n   dyntrace:restore_tag(Saved),\n   file:close(F).\n```\n\nIn this example, any user tag set in the calling process will be spread to the\nI/O-server when the `io:format/3` call is done.","ref":"dyntrace.html#spread_tag/1"},{"type":"type","title":"dyntrace.probe_arg/0","doc":"","ref":"dyntrace.html#t:probe_arg/0"},{"type":"module","title":"instrument","doc":"Analysis and Utility Functions for Instrumentation\n\nThe module `instrument` contains support for studying the resource usage in an\nErlang runtime system. Currently, only the allocation of memory can be studied.\n\n> #### Note {: .info }\n>\n> Since this module inspects internal details of the runtime system it may\n> differ greatly from one version to another. We make no compatibility\n> guarantees in this module.","ref":"instrument.html"},{"type":"module","title":"See Also - instrument","doc":"[erts_alloc(3)](`e:erts:erts_alloc.md`), [erl(1)](`e:erts:erl_cmd.md`)","ref":"instrument.html#module-see-also"},{"type":"function","title":"instrument.allocations/0","doc":"","ref":"instrument.html#allocations/0"},{"type":"function","title":"instrument.allocations/1","doc":"Returns a summary of all tagged allocations in the system, optionally filtered\nby allocator type and scheduler id.\n\nOnly binaries and allocations made by NIFs and drivers are tagged by default,\nbut this can be configured an a per-allocator basis with the\n[`+M atags` ](`e:erts:erts_alloc.md#M_atags`)emulator option.\n\nIf the specified allocator types are not enabled, the call will fail with\n`{error, not_enabled}`.\n\nThe following options can be used:\n\n- **`allocator_types`** - The allocator types that will be searched.\n\n  Specifying a specific allocator type may lead to strange results when carrier\n  migration between different allocator types has been enabled: you may see\n  unexpected types (e.g. process heaps when searching binary_alloc), or fewer\n  blocks than expected if the carriers the blocks are on have been migrated out\n  to an allocator of a different type.\n\n  Defaults to all `alloc_util` allocators.\n\n- **`scheduler_ids`** - The scheduler ids whose allocator instances will be\n  searched. A scheduler id of 0 will refer to the global instance that is not\n  tied to any particular scheduler. Defaults to all schedulers and the global\n  instance.\n\n- **`histogram_start`** - The upper bound of the first interval in the allocated\n  block size histograms. Defaults to 128.\n\n- **`histogram_width`** - The number of intervals in the allocated block size\n  histograms. Defaults to 18.\n\n- **`flags`** - Controls how to group the output, for example showing\n  allocations on a per-process basis (when possible) rather than only a\n  NIF/driver-basis. Defaults to `[]`.\n\n_Example:_\n\n```erlang\n> instrument:allocations(#{ histogram_start => 128, histogram_width => 15 }).\n{ok,{128,0,\n     #{udp_inet =>\n           #{driver_event_state => {0,0,0,0,0,0,0,0,0,1,0,0,0,0,0}},\n       system =>\n           #{heap => {0,0,0,0,20,4,2,2,2,3,0,1,0,0,1},\n             db_term => {271,3,1,52,80,1,0,0,0,0,0,0,0,0,0},\n             code => {0,0,0,5,3,6,11,22,19,20,10,2,1,0,0},\n             binary => {18,0,0,0,7,0,0,1,0,0,0,0,0,0,0},\n             message => {0,40,78,2,2,0,0,0,0,0,0,0,0,0,0},\n             ... }\n       spawn_forker =>\n           #{driver_select_data_state =>\n                 {1,0,0,0,0,0,0,0,0,0,0,0,0,0,0}},\n       ram_file_drv => #{drv_binary => {0,0,0,0,0,0,1,0,0,0,0,0,0,0,0}},\n       prim_file =>\n           #{process_specific_data => {2,0,0,0,0,0,0,0,0,0,0,0,0,0,0},\n             nif_trap_export_entry => {0,4,0,0,0,0,0,0,0,0,0,0,0,0,0},\n             monitor_extended => {0,1,0,0,0,0,0,0,0,0,0,0,0,0,0},\n             drv_binary => {0,0,0,0,0,0,1,0,3,5,0,0,0,1,0},\n             binary => {0,4,0,0,0,0,0,0,0,0,0,0,0,0,0}},\n       prim_buffer =>\n           #{nif_internal => {0,4,0,0,0,0,0,0,0,0,0,0,0,0,0},\n             binary => {0,4,0,0,0,0,0,0,0,0,0,0,0,0,0}}}}}\n```","ref":"instrument.html#allocations/1"},{"type":"function","title":"instrument.carriers/0","doc":"","ref":"instrument.html#carriers/0"},{"type":"function","title":"instrument.carriers/1","doc":"Returns a summary of all carriers in the system, optionally filtered by\nallocator type and scheduler id.\n\nIf the specified allocator types are not enabled, the call will fail with\n`{error, not_enabled}`.\n\nThe following options can be used:\n\n- **`allocator_types`** - The allocator types that will be searched. Defaults to\n  all `alloc_util` allocators.\n\n- **`scheduler_ids`** - The scheduler ids whose allocator instances will be\n  searched. A scheduler id of 0 will refer to the global instance that is not\n  tied to any particular scheduler. Defaults to all schedulers and the global\n  instance.\n\n- **`histogram_start`** - The upper bound of the first interval in the free\n  block size histograms. Defaults to 512.\n\n- **`histogram_width`** - The number of intervals in the free block size\n  histograms. Defaults to 14.\n\n_Example:_\n\n```erlang\n> instrument:carriers(#{ histogram_start => 512, histogram_width => 8 }).\n{ok,{512,\n     [{driver_alloc,false,262144,0,\n                    [{driver_alloc,1,32784}],\n                    {0,0,0,0,0,0,0,1}},\n      {binary_alloc,false,32768,0,\n                    [{binary_alloc,15,4304}],\n                    {3,0,0,0,1,0,0,0}},\n      {...}|...]}}\n```","ref":"instrument.html#carriers/1"},{"type":"type","title":"instrument.allocation_origin/0","doc":"","ref":"instrument.html#t:allocation_origin/0"},{"type":"type","title":"instrument.allocation_summary/0","doc":"A summary of allocated block sizes (including their headers) grouped by their\n`Origin` and `Type`.\n\n`Origin` is generally which NIF or driver that allocated the blocks, or 'system'\nif it could not be determined.\n\n`Type` is the allocation category that the blocks belong to, e.g. `db_term`,\n`message` or `binary`. The categories correspond to those in\n[erl_alloc.types](https://github.com/erlang/otp/blob/master/erts/emulator/beam/erl_alloc.types).\n\nIf one or more carriers could not be scanned in full without harming the\nresponsiveness of the system, `UnscannedSize` is the number of bytes that had to\nbe skipped.","ref":"instrument.html#t:allocation_summary/0"},{"type":"type","title":"instrument.block_histogram/0","doc":"A histogram of block sizes where each interval's upper bound is twice as high as\nthe one before it.\n\nThe upper bound of the first interval is provided by the function that returned\nthe histogram, and the last interval has no upper bound.\n\nFor example, the histogram below has 40 (`message`) blocks between 128-256 bytes\nin size, 78 blocks between 256-512 bytes,2 blocks between 512-1024 bytes, and 2\nblocks between 1-2KB.\n\n```erlang\n> instrument:allocations(#{ histogram_start => 128, histogram_width => 15 }).\n{ok, {128, 0, #{ message => {0,40,78,2,2,0,0,0,0,0,0,0,0,0,0}, ... } }}\n```","ref":"instrument.html#t:block_histogram/0"},{"type":"type","title":"instrument.carrier_info_list/0","doc":"`AllocatorType` is the type of the allocator that employs this carrier.\n\n`InPool` is whether the carrier is in the migration pool.\n\n`TotalSize` is the total size of the carrier, including its header.\n\n`Allocations` is a summary of the allocated blocks in the carrier. Note that\ncarriers may contain multiple different block types when carrier pools are\nshared between different allocator types (see the\n[`erts_alloc` ](`e:erts:erts_alloc.md#M_cp`)documentation for more details).\n\n`FreeBlocks` is a histogram of the free block sizes in the carrier.\n\nIf the carrier could not be scanned in full without harming the responsiveness\nof the system, `UnscannedSize` is the number of bytes that had to be skipped.","ref":"instrument.html#t:carrier_info_list/0"},{"type":"module","title":"msacc","doc":"Convenience functions for microstate accounting\n\nThis module implements some convenience functions for analyzing microstate\naccounting data. For details about how to use the basic API and what the\ndifferent states represent, see\n[`erlang:statistics(microstate_accounting)`](`m:erlang#statistics_microstate_accounting`).\n\n[](){: #msacc_print_example }\n\n_Basic Scenario_\n\n```erlang\n1> msacc:start(1000).\nok\n2> msacc:print().\nAverage thread real-time    : 1000513 us\nAccumulated system run-time :    2213 us\nAverage scheduler run-time  :    1076 us\n\n        Thread      aux check_io emulator       gc    other     port    sleep\n\nStats per thread:\n     async( 0)    0.00%    0.00%    0.00%    0.00%    0.00%    0.00%  100.00%\n     async( 1)    0.00%    0.00%    0.00%    0.00%    0.00%    0.00%  100.00%\n       aux( 1)    0.00%    0.00%    0.00%    0.00%    0.00%    0.00%   99.99%\n scheduler( 1)    0.00%    0.03%    0.13%    0.00%    0.01%    0.00%   99.82%\n scheduler( 2)    0.00%    0.00%    0.00%    0.00%    0.03%    0.00%   99.97%\n\nStats per type:\n         async    0.00%    0.00%    0.00%    0.00%    0.00%    0.00%  100.00%\n           aux    0.00%    0.00%    0.00%    0.00%    0.00%    0.00%   99.99%\n     scheduler    0.00%    0.02%    0.06%    0.00%    0.02%    0.00%   99.89%\nok\n```\n\nThis first command enables microstate accounting for 1000 milliseconds. See\n`start/0`, `stop/0`, `reset/0`, and `start/1` for more details. The second\ncommand prints the statistics gathered during that time. First three general\nstatistics are printed.\n\n- **Average real-time** - The average time spent collecting data in the threads.\n  This should be close to the time which data was collected.\n\n- **System run-time** - The total run-time of all threads in the system. This is\n  what you get if you call `msacc:stats(total_runtime,Stats).`\n\n- **Average scheduler run-time** - The average run-time for the schedulers. This\n  is the average amount of time the schedulers did not sleep.\n\nThen one column per state is printed with a the percentage of time this thread\nspent in the state out of it's own real-time. After the thread specific time,\nthe accumulated time for each type of thread is printed in a similar format.\n\nSince we have the average real-time and the percentage spent in each state we\ncan easily calculate the time spent in each state by multiplying\n`Average thread real-time` with `Thread state %`, that is, to get the time Scheduler\n1 spent in the emulator state we do `1000513us * 0.13% = 1300us`.","ref":"msacc.html"},{"type":"function","title":"msacc.available/0","doc":"This function checks whether microstate accounting is available or not.","ref":"msacc.html#available/0"},{"type":"function","title":"msacc.from_file/1","doc":"Read a file dump produced by [to_file(Filename)](`to_file/1`).","ref":"msacc.html#from_file/1"},{"type":"function","title":"msacc.print/0","doc":"Prints the current microstate accounting to standard out. Equivalent to\n[`msacc:print(msacc:stats(), #{}).`](`print/1`)","ref":"msacc.html#print/0"},{"type":"function","title":"msacc.print/1","doc":"","ref":"msacc.html#print/1"},{"type":"function","title":"msacc.print/2","doc":"Print the given microstate statistics values to standard out. With many states\nthis can be verbose. See the top of this reference manual for a brief\ndescription of what the fields mean.\n\nIt is possible to print more specific types of statistics by first manipulating\nthe `DataOrStats` using `stats/2`. For instance if you want to print the\npercentage of run-time for each thread you can do:\n\n```erlang\nmsacc:print(msacc:stats(runtime, msacc:stats())).\n```\n\nIf you want to only print run-time per thread type you can do:\n\n```erlang\nmsacc:print(msacc:stats(type, msacc:stats(runtime, msacc:stats()))).\n```\n\n_Options_\n\n- **`system`** - Print percentage of time spent in each state out of system time\n  as well as thread time. Default: false.","ref":"msacc.html#print/2"},{"type":"function","title":"msacc.print/3","doc":"Print the given microstate statistics values to the given file or device. The\nother arguments behave the same way as for `print/2`.","ref":"msacc.html#print/3"},{"type":"function","title":"msacc.reset/0","doc":"Reset microstate accounting counters. Returns whether is was enabled or\ndisabled.","ref":"msacc.html#reset/0"},{"type":"function","title":"msacc.start/0","doc":"Start microstate accounting. Returns whether it was previously enabled or\ndisabled.","ref":"msacc.html#start/0"},{"type":"function","title":"msacc.start/1","doc":"Resets all counters and then starts microstate accounting for the given\nmilliseconds.","ref":"msacc.html#start/1"},{"type":"function","title":"msacc.stats/0","doc":"Returns a runtime system independent version of the microstate statistics data\npresented by\n[`erlang:statistics(microstate_accounting)`](`m:erlang#statistics_microstate_accounting`).\nAll counters have been normalized to be in microsecond resolution.","ref":"msacc.html#stats/0"},{"type":"function","title":"msacc.stats/2","doc":"Returns the system time for the given microstate statistics values. System time\nis the accumulated time of all threads.\n\n- **`realtime`** - Returns all time recorded for all threads.\n\n- **`runtime`** - Returns all time spent doing work for all threads, i.e. all\n  time not spent in the `sleep` state.\n\nReturns fractions of real-time or run-time spent in the various threads from the\ngiven microstate statistics values.\n\nReturns a list of microstate statistics values where the values for all threads\nof the same type has been merged.","ref":"msacc.html#stats/2"},{"type":"function","title":"msacc.stop/0","doc":"Stop microstate accounting. Returns whether is was previously enabled or\ndisabled.","ref":"msacc.html#stop/0"},{"type":"function","title":"msacc.to_file/1","doc":"Dumps the current microstate statistics counters to a file that can be parsed\nwith `file:consult/1`.","ref":"msacc.html#to_file/1"},{"type":"type","title":"msacc.msacc_data/0","doc":"","ref":"msacc.html#t:msacc_data/0"},{"type":"type","title":"msacc.msacc_data_counters/0","doc":"A map containing the different microstate accounting states and the number of\nmicroseconds spent in it.","ref":"msacc.html#t:msacc_data_counters/0"},{"type":"type","title":"msacc.msacc_data_thread/0","doc":"","ref":"msacc.html#t:msacc_data_thread/0"},{"type":"type","title":"msacc.msacc_id/0","doc":"","ref":"msacc.html#t:msacc_id/0"},{"type":"type","title":"msacc.msacc_print_options/0","doc":"The different options that can be given to `print/2`.","ref":"msacc.html#t:msacc_print_options/0"},{"type":"type","title":"msacc.msacc_state/0","doc":"The different states that a thread can be in. See\n[erlang:statistics(microstate_accounting)](`m:erlang#statistics_microstate_accounting`)\nfor details.","ref":"msacc.html#t:msacc_state/0"},{"type":"type","title":"msacc.msacc_stats/0","doc":"","ref":"msacc.html#t:msacc_stats/0"},{"type":"type","title":"msacc.msacc_stats_counters/0","doc":"A map containing the different microstate accounting states. Each value in the\nmap contains another map with the percentage of time that this thread has spent\nin the specific state. Both the percentage of `system` time and the time for\nthat specific `thread` is part of the map.","ref":"msacc.html#t:msacc_stats_counters/0"},{"type":"type","title":"msacc.msacc_stats_thread/0","doc":"A map containing information about a specific thread. The percentages in the map\ncan be either run-time or real-time depending on if `runtime` or `realtime` was\nrequested from `stats/2`. `system` is the percentage of total system time for\nthis specific thread.","ref":"msacc.html#t:msacc_stats_thread/0"},{"type":"type","title":"msacc.msacc_type/0","doc":"","ref":"msacc.html#t:msacc_type/0"},{"type":"module","title":"scheduler","doc":"Measure scheduler utilization\n\nThis module contains utility functions for easy measurement and calculation of\nscheduler utilization. It act as a wrapper around the more primitive API\n[`erlang:statistics(scheduler_wall_time)`](`m:erlang#statistics_scheduler_wall_time`).\n\nThe simplest usage is to call the blocking\n[`scheduler:utilization(Seconds)`](`utilization/1`).\n\nFor non blocking and/or continuous calculation of scheduler utilization, the\nrecommended usage is:\n\n- First call\n  [`erlang:system_flag(scheduler_wall_time, true)`](`m:erlang#system_flag_scheduler_wall_time`)\n  to enable scheduler wall time measurements.\n- Call `get_sample/0` to collect samples with some time in between.\n- Call `utilization/2` to calculate the scheduler utilization in the interval\n  between two samples.\n- When done call\n  [`erlang:system_flag(scheduler_wall_time, false)`](`m:erlang#system_flag_scheduler_wall_time`)\n  to disable scheduler wall time measurements and avoid unecessary CPU overhead.\n\nTo get correct values from `utilization/2`, it is important that\n`scheduler_wall_time` is kept enabled during the entire interval between the two\nsamples. To ensure this, the process that called\n[`erlang:system_flag(scheduler_wall_time, true)`](`m:erlang#system_flag_scheduler_wall_time`)\nmust be kept alive, as `scheduler_wall_time` will automatically be disabled if\nit terminates.","ref":"scheduler.html"},{"type":"function","title":"scheduler.get_sample/0","doc":"Returns a scheduler utilization sample for normal and dirty-cpu schedulers.\nReturns `undefined` if system flag\n[`scheduler_wall_time`](`m:erlang#system_flag_scheduler_wall_time`) has not been\nenabled.","ref":"scheduler.html#get_sample/0"},{"type":"function","title":"scheduler.get_sample_all/0","doc":"Return a scheduler utilization sample for all schedulers, including dirty-io\nschedulers. Returns `undefined` if system flag\n[`scheduler_wall_time`](`m:erlang#system_flag_scheduler_wall_time`) has not been\nenabled.","ref":"scheduler.html#get_sample_all/0"},{"type":"function","title":"scheduler.sample/0","doc":"Return a scheduler utilization sample for normal and dirty-cpu schedulers. Will\ncall\n[`erlang:system_flag(scheduler_wall_time, true)`](`m:erlang#system_flag_scheduler_wall_time`)\nfirst if not already already enabled.\n\n> #### Note {: .info }\n>\n> This function is _not recommended_ as there is no way to detect if\n> `scheduler_wall_time` already was enabled or not. If `scheduler_wall_time` has\n> been disabled between two samples, passing them to\n> [`utilization/2`](`utilization/1`) will yield invalid results.\n>\n> Instead use `get_sample/0` together with\n> [`erlang:system_flag(scheduler_wall_time, _)`](`m:erlang#system_flag_scheduler_wall_time`).","ref":"scheduler.html#sample/0"},{"type":"function","title":"scheduler.sample_all/0","doc":"Return a scheduler utilization sample for all schedulers, including dirty-io\nschedulers. Will call\n[`erlang:system_flag(scheduler_wall_time, true)`](`m:erlang#system_flag_scheduler_wall_time`)\nfirst if not already already enabled.\n\n> #### Note {: .info }\n>\n> This function is _not recommended_ for same reason as `sample/0`. Instead use\n> `get_sample_all/0` together with\n> [`erlang:system_flag(scheduler_wall_time,_)`](`m:erlang#system_flag_scheduler_wall_time`).","ref":"scheduler.html#sample_all/0"},{"type":"function","title":"scheduler.utilization/1","doc":"Measure utilization for normal and dirty-cpu schedulers during `Seconds`\nseconds, and then return the result.\n\nWill automatically first enable and then disable\n[`scheduler_wall_time`](`m:erlang#system_flag_scheduler_wall_time`).\n\nCalculate scheduler utilizations for the time interval from when `Sample` was\ntaken and \"now\". The same as calling\n`scheduler:utilization(Sample, scheduler:sample_all())`.\n\n> #### Note {: .info }\n>\n> This function is _not recommended_ as it's so easy to get invalid results\n> without noticing. In particular do not do this:\n>\n> ```erlang\n> scheduler:utilization(scheduler:sample()). % DO NOT DO THIS!\n> ```\n>\n> The above example takes two samples in rapid succession and calculates the\n> scheduler utilization between them. The resulting values will probably be more\n> misleading than informative.\n>\n> Instead use [`scheduler:utilization/2`](`utilization/2`) and call\n> `get_sample/0` to get samples with some time in between.","ref":"scheduler.html#utilization/1"},{"type":"function","title":"scheduler.utilization/2","doc":"Calculates scheduler utilizations for the time interval between the two samples\nobtained from calling [`get_sample/0`](`sample/0`) or\n[`get_sample_all/0`](`sample_all/0`).\n\nThis function itself, does not need\n[`scheduler_wall_time`](`m:erlang#system_flag_scheduler_wall_time`) to be\nenabled. However, for a correct result, `scheduler_wall_time` must have been\nenabled during the entire interval between the two samples.","ref":"scheduler.html#utilization/2"},{"type":"type","title":"scheduler.sched_id/0","doc":"","ref":"scheduler.html#t:sched_id/0"},{"type":"opaque","title":"scheduler.sched_sample/0","doc":"","ref":"scheduler.html#t:sched_sample/0"},{"type":"type","title":"scheduler.sched_type/0","doc":"","ref":"scheduler.html#t:sched_type/0"},{"type":"type","title":"scheduler.sched_util_result/0","doc":"A list of tuples containing results for individual schedulers as well as\naggregated averages. `Util` is the scheduler utilization as a floating point\nvalue between 0.0 and 1.0. `Percent` is the same utilization as a more human\nreadable string expressed in percent.\n\n- **`{normal, SchedulerId, Util, Percent}`** - Scheduler utilization of a normal\n  scheduler with number `SchedulerId`. Schedulers that are not online will also\n  be included. [Online schedulers](`m:erlang#system_info_schedulers_online`)\n  have the lowest `SchedulerId`.\n\n- **`{cpu, SchedulerId, Util, Percent}`** - Scheduler utilization of a dirty-cpu\n  scheduler with number `SchedulerId`.\n\n- **`{io, SchedulerId, Util, Percent}`** - Scheduler utilization of a dirty-io\n  scheduler with number `SchedulerId`. This tuple will only exist if both\n  samples were taken with `sample_all/0`.\n\n- **`{total, Util, Percent}`** - Total utilization of all normal and dirty-cpu\n  schedulers.\n\n- **`{weighted, Util, Percent}`** - Total utilization of all normal and\n  dirty-cpu schedulers, weighted against maximum amount of available CPU time.","ref":"scheduler.html#t:sched_util_result/0"},{"type":"module","title":"system_information","doc":"System Information","ref":"system_information.html"},{"type":"function","title":"system_information.sanity_check/0","doc":"Performs a sanity check on the system.\n\nIf no issues were found, `ok` is returned. If issues were found,\n`{failed, Failures}` is returned. All failures found will be part of\nthe `Failures` list. Currently defined `Failure` elements in the\n`Failures` list:\n\n- **`InvalidAppFile`** - An application has an invalid `.app` file. The second\n  element identifies the application which has the invalid `.app` file.\n\n- **`InvalidApplicationVersion`** - An application has an invalid application\n  version. The second element identifies the application version that is\n  invalid.\n\n- **`MissingRuntimeDependencies`** - An application is missing\n  [runtime dependencies](`e:kernel:app.md#runtime_dependencies`). The second\n  element identifies the application (with version) that has missing\n  dependencies. The third element contains the missing dependencies.\n\n  Note that this check use application versions that are loaded, or will be\n  loaded when used. You might have application versions that satisfies all\n  dependencies installed in the system, but if those are not loaded this check\n  will fail. Of course, the system will also fail when used like this. This can\n  happen when you have multiple [branched versions](`e:system:versions.md`) of\n  the same application installed in the system, but there does not exist a\n  [boot script](`e:system:system_principles.md#BOOTSCRIPT`) identifying the\n  correct application version.\n\nCurrently the sanity check is limited to verifying runtime dependencies found in\nthe `.app` files of all applications. More checks will be introduced in the\nfuture. This implies that the return type _will_ change in the future.\n\n> #### Note {: .info }\n>\n> An `ok` return value only means that `sanity_check/0` did not find any issues,\n> _not_ that no issues exist.","ref":"system_information.html#sanity_check/0"},{"type":"function","title":"system_information.to_file/1","doc":"Writes miscellaneous system information to file. This information will typically\nbe requested by the Erlang/OTP team at Ericsson AB when reporting an issue.","ref":"system_information.html#to_file/1"},{"type":"extras","title":"Runtime tools Application","doc":"<!--\n%CopyrightBegin%\n\nCopyright Ericsson AB 2023-2024. All Rights Reserved.\n\nLicensed under the Apache License, Version 2.0 (the \"License\");\nyou may not use this file except in compliance with the License.\nYou may obtain a copy of the License at\n\n    http://www.apache.org/licenses/LICENSE-2.0\n\nUnless required by applicable law or agreed to in writing, software\ndistributed under the License is distributed on an \"AS IS\" BASIS,\nWITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.\nSee the License for the specific language governing permissions and\nlimitations under the License.\n\n%CopyrightEnd%\n-->\n# Runtime tools Application","ref":"runtime_tools_app.html"},{"type":"extras","title":"Description - Runtime tools Application","doc":"This chapter describes the Runtime_Tools application in OTP, which provides low\nfootprint tracing/debugging tools suitable for inclusion in a production system.","ref":"runtime_tools_app.html#description"},{"type":"extras","title":"Configuration - Runtime tools Application","doc":"There are currently no configuration parameters available for this application.","ref":"runtime_tools_app.html#configuration"},{"type":"extras","title":"SEE ALSO - Runtime tools Application","doc":"application(3)","ref":"runtime_tools_app.html#see-also"},{"type":"extras","title":"Runtime_Tools Release Notes","doc":"<!--\n%CopyrightBegin%\n\nCopyright Ericsson AB 2023-2024. All Rights Reserved.\n\nLicensed under the Apache License, Version 2.0 (the \"License\");\nyou may not use this file except in compliance with the License.\nYou may obtain a copy of the License at\n\n    http://www.apache.org/licenses/LICENSE-2.0\n\nUnless required by applicable law or agreed to in writing, software\ndistributed under the License is distributed on an \"AS IS\" BASIS,\nWITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.\nSee the License for the specific language governing permissions and\nlimitations under the License.\n\n%CopyrightEnd%\n-->\n# Runtime_Tools Release Notes\n\nThis document describes the changes made to the Runtime_Tools application.","ref":"notes.html"},{"type":"extras","title":"Runtime_Tools 2.1 - Runtime_Tools Release Notes","doc":"","ref":"notes.html#runtime_tools-2-1"},{"type":"extras","title":"Improvements and New Features - Runtime_Tools Release Notes","doc":"- The `m:instrument` module can now track allocations on a per-process or per-port basis.\n\n  Own Id: OTP-18577 Aux Id: [PR-7236]\n\n- The new function `proc_lib:set_label/1` can be used to add a descriptive term to any process that does not have a registered name. The name will be shown by tools such as `\\c:i/0`, `m:observer`, and it will be included in crash reports produced by processes using `m:gen_server`, `m:gen_statem`, `m:gen_event`, and `m:gen_fsm`.\n  \n  The label for a process can be retrieved by calling `proc_lib:get_label/1`.\n  \n  Note that those functions work on any process, not only processes that use `m:proc_lib`.\n  \n  Example:\n  \n  ```text\n  1> self().\n  <0.90.0>\n  2> proc_lib:set_label(my_label).\n  ok\n  3> i().\n      .\n      .\n      .\n  <0.90.0>              erlang:apply/2                        2586    75011    0\n  my_label              c:pinfo/2                               51\n  4> proc_lib:get_label(self()).\n  my_label\n  ```\n\n  Own Id: OTP-18789 Aux Id: [PR-7720], [PR-8003]\n\n- Type specs had been added to all `m:dbg` functions.\n\n  Own Id: OTP-18859 Aux Id: [PR-7782]\n\n- The documentation has been migrated to use Markdown and ExDoc.\n\n  Own Id: OTP-18955 Aux Id: [PR-8026]\n\n- `m:dbg` have been updated to use [trace sessions](`t:trace:session/0`).\n  \n  `dbg:session_create/1`, `dbg:session/2`, and `dbg:session_destroy/1` have been added to work sessions. See the documentation for details.\n\n  Own Id: OTP-19081 Aux Id: [PR-8363]\n\n[PR-7236]: https://github.com/erlang/otp/pull/7236\n[PR-7720]: https://github.com/erlang/otp/pull/7720\n[PR-8003]: https://github.com/erlang/otp/pull/8003\n[PR-7782]: https://github.com/erlang/otp/pull/7782\n[PR-8026]: https://github.com/erlang/otp/pull/8026\n[PR-8363]: https://github.com/erlang/otp/pull/8363","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Runtime_Tools 2.0.1 - Runtime_Tools Release Notes","doc":"","ref":"notes.html#runtime_tools-2-0-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Runtime_Tools Release Notes","doc":"- Fixed issue with fetching port information for observer could crash if port\n  had died.\n\n  Own Id: OTP-18868 Aux Id: GH-7735\n\n- Fixed some benign compile warnings on Windows.\n\n  Own Id: OTP-18895","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Runtime_Tools 2.0 - Runtime_Tools Release Notes","doc":"","ref":"notes.html#runtime_tools-2-0"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Runtime_Tools Release Notes","doc":"- Fixed the type specification for `instrument:carriers/0,1`\n\n  Own Id: OTP-18499 Aux Id: PR-6946","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Runtime_Tools Release Notes","doc":"- Add `dbg:tracer(file, Filename)` as a convenient way to trace to a file in\n  clean text.\n\n  Own Id: OTP-18211 Aux Id: PR-6143\n\n- Handling of `on_load` modules during boot has been improved by adding an extra\n  step in the boot order for embedded mode that runs all `on_load` handlers,\n  instead of relying on explicit invocation of them, later, when the kernel\n  supervision tree starts.\n\n  This is mostly a code improvement and OTP internal simplification to avoid\n  future bugs and to simplify code maintenance.\n\n  Own Id: OTP-18447\n\n- Deprecates `dbg:stop_clear/0` because it is simply a function alias to\n  `dbg:stop/0`\n\n  Own Id: OTP-18478 Aux Id: GH-6903\n\n- The `instrument` module has been moved from `tools` to `runtime_tools`.\n\n  Own Id: OTP-18487 Aux Id: PR-6829\n\n- Removed the experimental `erts_alloc_config` module. It no longer produced\n  good configurations and cannot be fixed in a reasonably backwards compatible\n  manner. It has since OTP 25 been deprecated and scheduled for removal in\n  OTP 26.\n\n  \\*** POTENTIAL INCOMPATIBILITY \\***\n\n  Own Id: OTP-18549 Aux Id: PR-7105","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Runtime_Tools 1.19 - Runtime_Tools Release Notes","doc":"","ref":"notes.html#runtime_tools-1-19"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Runtime_Tools Release Notes","doc":"- Reading port socket options on macOS and Windows \"skips\" invalid options.\n\n  Own Id: OTP-18012 Aux Id: #5798","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Runtime_Tools Release Notes","doc":"- `dbg:stop/0` now behaves like `dbg:stop_clear/0`, clearing all global trace\n  patterns for all functions.\n\n  Own Id: OTP-17909 Aux Id: ERIERL-760\n\n- `erts_alloc_config` has been scheduled for removal in OTP 26. It has not\n  produced good configurations for a very long time, and unfortunately it cannot\n  be fixed in a backwards compatible manner.\n\n  Own Id: OTP-17939","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Runtime_Tools 1.18 - Runtime_Tools Release Notes","doc":"","ref":"notes.html#runtime_tools-1-18"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Runtime_Tools Release Notes","doc":"- Fixed bug in `scheduler:utilization(Seconds)` that would leave the\n  `scheduler_wall_time` system flag incorrectly enabled.\n\n  Own Id: OTP-17800 Aux Id: PR-5425","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Runtime_Tools Release Notes","doc":"- Add `scheduler:get_sample/0` and `get_sample_all/0`. Also clarify `scheduler`\n  module documentation about how it depends on system flag\n  `scheduler_wall_time`.\n\n  Own Id: OTP-17830 Aux Id: GH-5425, PR-5444","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Runtime_Tools 1.17 - Runtime_Tools Release Notes","doc":"","ref":"notes.html#runtime_tools-1-17"},{"type":"extras","title":"Improvements and New Features - Runtime_Tools Release Notes","doc":"- Observer now has a sectiion for new socket.\n\n  Own Id: OTP-17346\n\n- The dbg docs have been expanded to include the meaning of all the function\n  name acronyms.\n\n  Own Id: OTP-17572 Aux Id: PR-5117","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Runtime_Tools 1.16.2 - Runtime_Tools Release Notes","doc":"","ref":"notes.html#runtime_tools-1-16-2"},{"type":"extras","title":"Improvements and New Features - Runtime_Tools Release Notes","doc":"- The experimental HiPE application has been removed, together with all related\n  functionality in other applications.\n\n  \\*** POTENTIAL INCOMPATIBILITY \\***\n\n  Own Id: OTP-16963","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Runtime_Tools 1.16.1 - Runtime_Tools Release Notes","doc":"","ref":"notes.html#runtime_tools-1-16-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Runtime_Tools Release Notes","doc":"- The function `dbg:n/1` used a local fun to set up a tracer on a remote node.\n  This works fine as long as the remote node is running exactly the same version\n  of Erlang/OTP but does not work at all otherwise. This is fixed by exporting\n  the relevant function and by calling this function on the remote node to set\n  up remote tracing.\n\n  Own Id: OTP-16930 Aux Id: ERL-1371, GH-4396","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Runtime_Tools 1.16 - Runtime_Tools Release Notes","doc":"","ref":"notes.html#runtime_tools-1-16"},{"type":"extras","title":"Improvements and New Features - Runtime_Tools Release Notes","doc":"- Clarify documentation of module 'scheduler'.\n\n  Own Id: OTP-17208 Aux Id: GH-4502, PR-4532","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Runtime_Tools 1.15.1 - Runtime_Tools Release Notes","doc":"","ref":"notes.html#runtime_tools-1-15-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Runtime_Tools Release Notes","doc":"- Fixed a crash in `appmon_info` triggered by trying to read port info from a\n  port that was in the process of terminating.\n\n  `appmon_info` is used by `observer` to get information from the observed node.\n\n  Own Id: OTP-16787 Aux Id: PR-2673","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Runtime_Tools 1.15 - Runtime_Tools Release Notes","doc":"","ref":"notes.html#runtime_tools-1-15"},{"type":"extras","title":"Improvements and New Features - Runtime_Tools Release Notes","doc":"- Improved the presentation of allocations and carriers in the `instrument`\n  module.\n\n  Own Id: OTP-16327\n\n- Minor updates due to the new spawn improvements made.\n\n  Own Id: OTP-16368 Aux Id: OTP-15251","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Runtime_Tools 1.14.0.1 - Runtime_Tools Release Notes","doc":"","ref":"notes.html#runtime_tools-1-14-0-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Runtime_Tools Release Notes","doc":"- The function `dbg:n/1` used a local fun to set up a tracer on a remote node.\n  This works fine as long as the remote node is running exactly the same version\n  of Erlang/OTP but does not work at all otherwise. This is fixed by exporting\n  the relevant function and by calling this function on the remote node to set\n  up remote tracing.\n\n  Own Id: OTP-16930 Aux Id: ERL-1371, GH-4396","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Runtime_Tools 1.14 - Runtime_Tools Release Notes","doc":"","ref":"notes.html#runtime_tools-1-14"},{"type":"extras","title":"Improvements and New Features - Runtime_Tools Release Notes","doc":"- Fix `dbg:stop_clear/0` to also clear trace events (`send` and `'receive'`).\n\n  Own Id: OTP-16044","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Runtime_Tools 1.13.3 - Runtime_Tools Release Notes","doc":"","ref":"notes.html#runtime_tools-1-13-3"},{"type":"extras","title":"Improvements and New Features - Runtime_Tools Release Notes","doc":"- Minor updates to build system necessary due to move of configuration of\n  `crypto` application.\n\n  Own Id: OTP-15262 Aux Id: OTP-15129","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Runtime_Tools 1.13.2.1 - Runtime_Tools Release Notes","doc":"","ref":"notes.html#runtime_tools-1-13-2-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Runtime_Tools Release Notes","doc":"- The function `dbg:n/1` used a local fun to set up a tracer on a remote node.\n  This works fine as long as the remote node is running exactly the same version\n  of Erlang/OTP but does not work at all otherwise. This is fixed by exporting\n  the relevant function and by calling this function on the remote node to set\n  up remote tracing.\n\n  Own Id: OTP-16930 Aux Id: ERL-1371, GH-4396","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Runtime_Tools 1.13.2 - Runtime_Tools Release Notes","doc":"","ref":"notes.html#runtime_tools-1-13-2"},{"type":"extras","title":"Improvements and New Features - Runtime_Tools Release Notes","doc":"- Update of systemtap trace example scripts.\n\n  Own Id: OTP-15670","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Runtime_Tools 1.13.1 - Runtime_Tools Release Notes","doc":"","ref":"notes.html#runtime_tools-1-13-1"},{"type":"extras","title":"Improvements and New Features - Runtime_Tools Release Notes","doc":"- Optimize `observer` by using new `system_info(ets_count)` instead of more\n  expensive [`length(ets:all())`](`length/1`).\n\n  Own Id: OTP-15163 Aux Id: PR-1844","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Runtime_Tools 1.13 - Runtime_Tools Release Notes","doc":"","ref":"notes.html#runtime_tools-1-13"},{"type":"extras","title":"Improvements and New Features - Runtime_Tools Release Notes","doc":"- New utility module `scheduler` which makes it easier to measure scheduler\n  utilization.\n\n  Own Id: OTP-14904","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Runtime_Tools 1.12.5 - Runtime_Tools Release Notes","doc":"","ref":"notes.html#runtime_tools-1-12-5"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Runtime_Tools Release Notes","doc":"- `system_information:to_file/1` will now use slightly less memory.\n\n  Own Id: OTP-14816","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Runtime_Tools 1.12.4 - Runtime_Tools Release Notes","doc":"","ref":"notes.html#runtime_tools-1-12-4"},{"type":"extras","title":"Improvements and New Features - Runtime_Tools Release Notes","doc":"- New family of `erts_alloc` strategies: Age Order First Fit. Similar to\n  \"address order\", but instead the oldest possible carrier is always chosen for\n  allocation.\n\n  Own Id: OTP-14917 Aux Id: ERIERL-88","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Runtime_Tools 1.12.3 - Runtime_Tools Release Notes","doc":"","ref":"notes.html#runtime_tools-1-12-3"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Runtime_Tools Release Notes","doc":"- Removed all old unused files in the documentation.\n\n  Own Id: OTP-14475 Aux Id: ERL-409, PR-1493","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Runtime_Tools 1.12.2 - Runtime_Tools Release Notes","doc":"","ref":"notes.html#runtime_tools-1-12-2"},{"type":"extras","title":"Improvements and New Features - Runtime_Tools Release Notes","doc":"- General Unicode improvements.\n\n  Own Id: OTP-14462","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Runtime_Tools 1.12.1 - Runtime_Tools Release Notes","doc":"","ref":"notes.html#runtime_tools-1-12-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Runtime_Tools Release Notes","doc":"- A faulty encoding comment was added when saving trace patterns to file. This\n  is now corrected.\n\n  Own Id: OTP-14479","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Runtime_Tools 1.12 - Runtime_Tools Release Notes","doc":"","ref":"notes.html#runtime_tools-1-12"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Runtime_Tools Release Notes","doc":"- Add compile option `-compile(no_native)` in modules with `on_load` directive\n  which is not yet supported by HiPE.\n\n  Own Id: OTP-14316 Aux Id: PR-1390","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Runtime_Tools Release Notes","doc":"- Miscellaneous updates due to atoms containing arbitrary Unicode characters.\n\n  Own Id: OTP-14285\n\n- Sockets can now be bound to device (SO_BINDTODEVICE) on platforms where it is\n  supported.\n\n  This has been implemented e.g to support VRF-Lite under Linux; see\n  [VRF ](https://www.kernel.org/doc/Documentation/networking/vrf.txt), and\n  GitHub pull request [\\#1326](https://github.com/erlang/otp/pull/1326).\n\n  Own Id: OTP-14357 Aux Id: PR-1326","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Runtime_Tools 1.11.1 - Runtime_Tools Release Notes","doc":"","ref":"notes.html#runtime_tools-1-11-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Runtime_Tools Release Notes","doc":"- etop erroneously reported the average scheduler utilization since the tool was\n  first started instead of the scheduler utilization since last update. This is\n  now corrected.\n\n  Own Id: OTP-14090 Aux Id: seq13232","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Runtime_Tools 1.11 - Runtime_Tools Release Notes","doc":"","ref":"notes.html#runtime_tools-1-11"},{"type":"extras","title":"Improvements and New Features - Runtime_Tools Release Notes","doc":"- Add option `queue_size` to ttb:tracer/2. This sets the maximum queue size for\n  the IP trace driver which is used when tracing to shell and/or `{local,File}`.\n\n  The default value for `queue_size` is specified by `dbg`, and it is now\n  changed from 50 to 200.\n\n  Own Id: OTP-13829 Aux Id: seq13171\n\n- The port information page is updated to show more information per port.\n\n  Own Id: OTP-13948 Aux Id: ERL-272","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Runtime_Tools 1.10.1 - Runtime_Tools Release Notes","doc":"","ref":"notes.html#runtime_tools-1-10-1"},{"type":"extras","title":"Improvements and New Features - Runtime_Tools Release Notes","doc":"- Correct some minor documentation issues.\n\n  Own Id: OTP-13891","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Runtime_Tools 1.10 - Runtime_Tools Release Notes","doc":"","ref":"notes.html#runtime_tools-1-10"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Runtime_Tools Release Notes","doc":"- Fix bug in dbg:trace_port/2 that could cause the trace ip driver to produce\n  faulty error reports \"...(re)selected before stop_select was called for driver\n  trace_ip_drv\".\n\n  Own Id: OTP-13576 Aux Id: ERL-119","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Runtime_Tools Release Notes","doc":"- Add microstate accounting\n\n  Microstate accounting is a way to track which state the different threads\n  within ERTS are in. The main usage area is to pin point performance\n  bottlenecks by checking which states the threads are in and then from there\n  figuring out why and where to optimize.\n\n  Since checking whether microstate accounting is on or off is relatively\n  expensive only a few of the states are enabled by default and more states can\n  be enabled through configure.\n\n  There is a convenience module called msacc that has been added to\n  runtime_tools that can assist in gathering and interpreting the data from\n  Microstate accounting.\n\n  For more information see\n  [erlang:statistics(microstate*accounting, *)](`m:erlang#statistics_microstate_accounting`)\n  and the `m:msacc` module in runtime_tools.\n\n  Own Id: OTP-12345\n\n- Update observer GUI to support tracing on ports, and to set matchspecs for\n  send/receive. This required some minor bugfixes in runtime_tools/dbg.\n\n  Own Id: OTP-13481\n\n- Update dbg and ttb to work with a tracer module as tracer and tracing on\n  ports.\n\n  Own Id: OTP-13500\n\n- Updated dbg to accept the new trace options `monotonic_timestamp` and\n  `strict_monotonic_timestamp`.\n\n  Own Id: OTP-13502\n\n- Introduce LTTng tracing via Erlang tracing.\n\n  For LTTng to be enabled OTP needs to be built with configure option\n  `--with-dynamic-trace=lttng`.\n\n  The dynamic trace module `dyntrace` is now capable to be used as a LTTng sink\n  for Erlang tracing. For a list of all tracepoints, see\n  [Runtime Tools User's Guide](lttng.md) .\n\n  This feature also introduces an incompatible change in trace tags. The trace\n  tags `gc_start` and `gc_end` has been split into `gc_minor_start`,\n  `gc_minor_end` and `gc_major_start`, `gc_major_end`.\n\n  \\*** POTENTIAL INCOMPATIBILITY \\***\n\n  Own Id: OTP-13532","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Runtime_Tools 1.9.3 - Runtime_Tools Release Notes","doc":"","ref":"notes.html#runtime_tools-1-9-3"},{"type":"extras","title":"Improvements and New Features - Runtime_Tools Release Notes","doc":"- `dbg:trace_client()` now uses a read buffer to speed up reading of trace\n  files.\n\n  Own Id: OTP-13279","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Runtime_Tools 1.9.2 - Runtime_Tools Release Notes","doc":"","ref":"notes.html#runtime_tools-1-9-2"},{"type":"extras","title":"Improvements and New Features - Runtime_Tools Release Notes","doc":"- Clarified dbg:stop documentation\n\n  Own Id: OTP-13078","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Runtime_Tools 1.9.1 - Runtime_Tools Release Notes","doc":"","ref":"notes.html#runtime_tools-1-9-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Runtime_Tools Release Notes","doc":"- The `trace_file_drv` did not handle `EINTR` correct which caused it to fail\n  when the runtime system received a signal.\n\n  Own Id: OTP-12890 Aux Id: seq12885","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Runtime_Tools 1.9 - Runtime_Tools Release Notes","doc":"","ref":"notes.html#runtime_tools-1-9"},{"type":"extras","title":"Improvements and New Features - Runtime_Tools Release Notes","doc":"- Change license text from Erlang Public License to Apache Public License v2\n\n  Own Id: OTP-12845","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Runtime_Tools 1.8.16 - Runtime_Tools Release Notes","doc":"","ref":"notes.html#runtime_tools-1-8-16"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Runtime_Tools Release Notes","doc":"- The trace process started by `dbg` would not always terminate when\n  `dbg:stop/0` was called.\n\n  Own Id: OTP-12517","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Runtime_Tools 1.8.15 - Runtime_Tools Release Notes","doc":"","ref":"notes.html#runtime_tools-1-8-15"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Runtime_Tools Release Notes","doc":"- Add `nif_version` to `erlang:system_info/1` in order to get the NIF API\n  version of the runtime system in a way similar to `driver_version`.\n\n  Own Id: OTP-12298","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Runtime_Tools 1.8.14 - Runtime_Tools Release Notes","doc":"","ref":"notes.html#runtime_tools-1-8-14"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Runtime_Tools Release Notes","doc":"- The documentation for the return value of dbg:\\{stop,stop_clear\\} functions\n  are now correct (Thanks to Luca Favatella)\n\n  Own Id: OTP-11603\n\n- Fix DTrace build on Illumos. (Thanks to Ryan Zezeski.)\n\n  Own Id: OTP-11622\n\n- Do not turn off scheduler_wall_time, as it can interfere with other\n  applications usage.\n\n  Own Id: OTP-11693 Aux Id: seq12528\n\n- Application upgrade (appup) files are corrected for the following\n  applications:\n\n  `asn1, common_test, compiler, crypto, debugger, dialyzer, edoc, eldap, erl_docgen, et, eunit, gs, hipe, inets, observer, odbc, os_mon, otp_mibs, parsetools, percept, public_key, reltool, runtime_tools, ssh, syntax_tools, test_server, tools, typer, webtool, wx, xmerl`\n\n  A new test utility for testing appup files is added to test_server. This is\n  now used by most applications in OTP.\n\n  (Thanks to Tobias Schlager)\n\n  Own Id: OTP-11744","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Runtime_Tools Release Notes","doc":"- Allow install path to have unicode characters.\n\n  Own Id: OTP-10877\n\n- The `erts_alloc_config` tool has been updated to produce configurations that\n  better fit todays SMP support in the VM.\n\n  Own Id: OTP-11662\n\n- The [`app`](`e:kernel:app.md`)\\-file key\n  [`runtime_dependencies`](`e:kernel:app.md#runtime_dependencies`) has been\n  introduced.\n\n  Runtime dependencies have been added to all app-files in OTP. Note that these\n  may not be completely correct during OTP 17, but this is actively being worked\n  on.\n\n  The function `system_information:sanity_check/0` will verify all declared\n  runtime dependencies in the system when called.\n\n  Own Id: OTP-11773","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Runtime_Tools 1.8.13 - Runtime_Tools Release Notes","doc":"","ref":"notes.html#runtime_tools-1-8-13"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Runtime_Tools Release Notes","doc":"- Observer did not produce correct result when ERTS internal memory allocators\n  had been disabled.\n\n  Own Id: OTP-11520","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Runtime_Tools 1.8.12 - Runtime_Tools Release Notes","doc":"","ref":"notes.html#runtime_tools-1-8-12"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Runtime_Tools Release Notes","doc":"- The process trace flag 'silent' is now allowed in call to dbg:p/2.\n\n  Own Id: OTP-11222","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Runtime_Tools Release Notes","doc":"- Introduced functionality for inspection of system and build configuration.\n\n  Own Id: OTP-11196","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Runtime_Tools 1.8.11 - Runtime_Tools Release Notes","doc":"","ref":"notes.html#runtime_tools-1-8-11"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Runtime_Tools Release Notes","doc":"- Some bugs related to calculation of CPU/scheduler utilization in observer are\n  corrected.\n\n  Current function for a process is accepted to be 'undefined' when running\n  hipe.\n\n  Own Id: OTP-10894","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Runtime_Tools Release Notes","doc":"- Erlang source files with non-ASCII characters are now encoded in UTF-8\n  (instead of latin1).\n\n  Own Id: OTP-11041 Aux Id: OTP-10907","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Runtime_Tools 1.8.10 - Runtime_Tools Release Notes","doc":"","ref":"notes.html#runtime_tools-1-8-10"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Runtime_Tools Release Notes","doc":"- Fix Table Viewer refresh crash on no more existing ets tables (Thanks to Peti\n  Gömori)\n\n  Own Id: OTP-10635","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Runtime_Tools Release Notes","doc":"- User Guides for the dynamic tracing tools dtrace and systemtap have been added\n  to the documentation.\n\n  Own Id: OTP-10155\n\n- Where necessary a comment stating encoding has been added to Erlang files. The\n  comment is meant to be removed in Erlang/OTP R17B when UTF-8 becomes the\n  default encoding.\n\n  Own Id: OTP-10630\n\n- Some examples overflowing the width of PDF pages have been corrected.\n\n  Own Id: OTP-10665\n\n- The backend module appmon_info.erl is moved from appmon application to\n  runtime_tools. This allows appmon to be run from a remote erlang node towards\n  a target node which does not have appmon (and its dependencies) installed, as\n  long as runtime_tools is installed there.\n\n  Own Id: OTP-10786","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Runtime_Tools 1.8.9 - Runtime_Tools Release Notes","doc":"","ref":"notes.html#runtime_tools-1-8-9"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Runtime_Tools Release Notes","doc":"- Change the module-level docs to give complete step-by-step instructions for\n  using the \\`dyntrace:p()\\` trace function. (Thanks to Scott Lystig Fritchie)\n\n  Own Id: OTP-10141\n\n- Add 1024 separate USDT probes to dyntrace.erl and dyntrace.c (Thanks to Scott\n  Lystig Fritchie)\n\n  Own Id: OTP-10143\n\n- Relocate bodies of DTrace probes to the statically-linked VM.\n\n  Due to various operating systems (in both the DTrace and SystemTap worlds) not\n  fully supporting DTrace probes (or SystemTap-compatibility mode probes) in\n  shared libraries, we relocate those probes to the statically-linked virtual\n  machine. This could be seen as pollution of the pristine VM by a (yet)\n  experimental feature. However:\n\n  1\\. This code can be eliminated completely by the C preprocessor. 2. Leaving\n  the probes in the dyntrace NIF shared library simply does not work correctly\n  on too many platforms. _Many_ thanks to Macneil Shonle at Basho for assisting\n  when my RSI-injured fingers gave out. (note: Solaris 10 and FreeBSD\n  9.0-RELEASE can take a long time to compile)\n\n  Own Id: OTP-10189","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Runtime_Tools 1.8.8 - Runtime_Tools Release Notes","doc":"","ref":"notes.html#runtime_tools-1-8-8"},{"type":"extras","title":"Improvements and New Features - Runtime_Tools Release Notes","doc":"- The DTrace source patch from Scott Lystig Fritchie is integrated in the source\n  tree. Using an emulator with dtrace probe is still not supported for\n  production use, but may be a valuable debugging tool. Configure with\n  --with-dynamic-trace=dtrace (or --with-dynamic-trace=systemtap) to create a\n  build with dtrace probes enabled. See runtime_tools for documentation and\n  examples.\n\n  Own Id: OTP-10017","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Runtime_Tools 1.8.7 - Runtime_Tools Release Notes","doc":"","ref":"notes.html#runtime_tools-1-8-7"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Runtime_Tools Release Notes","doc":"- Earlier dbg:stop only did erlang:trace_delivered and did not flush the trace\n  file driver. Therefore there could still be trace messages that were delivered\n  to the driver (guaranteed by erlang:trace_delivered) but not yet written to\n  the file when dbg:stop returned. Flushing is now added on each node before the\n  dbg process terminates.\n\n  Own Id: OTP-9651\n\n- File handles created by the trace_file_drv driver was inherited to child\n  processes. This is now corrected.\n\n  Own Id: OTP-9658","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Runtime_Tools Release Notes","doc":"- Erlang/OTP can now be built using parallel make if you limit the number of\n  jobs, for instance using '`make -j6`' or '`make -j10`'. '`make -j`' does not\n  work at the moment because of some missing dependencies.\n\n  Own Id: OTP-9451\n\n- Two new built-in trace pattern aliases have been added: caller_trace (c) and\n  caller_exception_trace (cx). See the dbg:ltp/0 documentation for more info.\n\n  Own Id: OTP-9458","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Runtime_Tools 1.8.6 - Runtime_Tools Release Notes","doc":"","ref":"notes.html#runtime_tools-1-8-6"},{"type":"extras","title":"Improvements and New Features - Runtime_Tools Release Notes","doc":"- Two new built-in trace pattern aliases have been added: caller_trace (c) and\n  caller_exception_trace (cx). See the dbg:ltp/0 documentation for more info.\n\n  Own Id: OTP-9458","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Runtime_Tools 1.8.5 - Runtime_Tools Release Notes","doc":"","ref":"notes.html#runtime_tools-1-8-5"},{"type":"extras","title":"Improvements and New Features - Runtime_Tools Release Notes","doc":"- When a big number of trace patterns are set by inviso the Erlang VM could get\n  unresponsive for several seconds. This is now corrected.\n\n  Own Id: OTP-9048 Aux Id: seq11480","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Runtime_Tools 1.8.4.1 - Runtime_Tools Release Notes","doc":"","ref":"notes.html#runtime_tools-1-8-4-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Runtime_Tools Release Notes","doc":"- Minor corrections and removal of a temporary workaround.\n\n  Own Id: OTP-8755 Aux Id: seq-11628, seq-11639\n\n- Small fix in inviso_autostart_server.\n\n  Own Id: OTP-8783 Aux Id: seq11628","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Runtime_Tools 1.8.4 - Runtime_Tools Release Notes","doc":"","ref":"notes.html#runtime_tools-1-8-4"},{"type":"extras","title":"Improvements and New Features - Runtime_Tools Release Notes","doc":"- Miscellaneous updates.\n\n  Own Id: OTP-8705","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Runtime_Tools 1.8.3 - Runtime_Tools Release Notes","doc":"","ref":"notes.html#runtime_tools-1-8-3"},{"type":"extras","title":"Improvements and New Features - Runtime_Tools Release Notes","doc":"- Cross compilation improvements and other build system improvements.\n\n  Most notable:\n\n  - Lots of cross compilation improvements. The old cross compilation support\n    was more or less non-existing as well as broken. Please, note that the cross\n    compilation support should still be considered as experimental. Also note\n    that old cross compilation configurations cannot be used without\n    modifications. For more information on cross compiling Erlang/OTP see the\n    `$ERL_TOP/INSTALL-CROSS.md` file.\n  - Support for staged install using\n    [DESTDIR](http://www.gnu.org/prep/standards/html_node/DESTDIR.html). The old\n    broken `INSTALL_PREFIX` has also been fixed. For more information see the\n    `$ERL_TOP/INSTALL.md` file.\n  - Documentation of the `release` target of the top `Makefile`. For more\n    information see the `$ERL_TOP/INSTALL.md` file.\n  - `make install` now by default creates relative symbolic links instead of\n    absolute ones. For more information see the `$ERL_TOP/INSTALL.md` file.\n  - `$ERL_TOP/configure --help=recursive` now works and prints help for all\n    applications with `configure` scripts.\n  - Doing `make install`, or `make release` directly after `make all` no longer\n    triggers miscellaneous rebuilds.\n  - Existing bootstrap system is now used when doing `make install`, or\n    `make release` without a preceding `make all`.\n  - The `crypto` and `ssl` applications use the same runtime library path when\n    dynamically linking against `libssl.so` and `libcrypto.so`. The runtime\n    library search path has also been extended.\n  - The `configure` scripts of `erl_interface` and `odbc` now search for thread\n    libraries and thread library quirks the same way as ERTS do.\n  - The `configure` script of the `odbc` application now also looks for odbc\n    libraries in `lib64` and `lib/64` directories when building on a 64-bit\n    system.\n  - The `config.h.in` file in the `erl_interface` application is now\n    automatically generated in instead of statically updated which reduces the\n    risk of `configure` tests without any effect.\n\n  (Thanks to Henrik Riomar for suggestions and testing)\n\n  (Thanks to Winston Smith for the AVR32-Linux cross configuration and testing)\n\n  \\*** POTENTIAL INCOMPATIBILITY \\***\n\n  Own Id: OTP-8323\n\n- Cleanups suggested by tidier and modernization of types and specs.\n\n  Own Id: OTP-8455","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Runtime_Tools 1.8.2 - Runtime_Tools Release Notes","doc":"","ref":"notes.html#runtime_tools-1-8-2"},{"type":"extras","title":"Improvements and New Features - Runtime_Tools Release Notes","doc":"- The documentation is now built with open source tools (xsltproc and fop) that\n  exists on most platforms. One visible change is that the frames are removed.\n\n  Own Id: OTP-8201","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Runtime_Tools 1.8.1 - Runtime_Tools Release Notes","doc":"","ref":"notes.html#runtime_tools-1-8-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Runtime_Tools Release Notes","doc":"- `Makefile.in` has been updated to use the LDFLAGS environment variable (if\n  set). (Thanks to Davide Pesavento.)\n\n  Own Id: OTP-8157","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Runtime_Tools 1.8 - Runtime_Tools Release Notes","doc":"","ref":"notes.html#runtime_tools-1-8"},{"type":"extras","title":"Improvements and New Features - Runtime_Tools Release Notes","doc":"- `etop` would crash if the emulator's custom allocators had been turned off\n  (e.g. using the `+Meamin` option).\n\n  Own Id: OTP-7519\n\n- The copyright notices have been updated.\n\n  Own Id: OTP-7851\n\n- Now, dbg:p/2 accepts \\{X,Y,Z\\} process specification as stated in the\n  documentation. It also now accepts \" \" like from erlang:pid_to_list/1.\n\n  There is now a pre-saved match spec in dbg that saves a lot of typing. Use\n  dbg:ltp/0 to find out more...\n\n  Own Id: OTP-7867","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Runtime_Tools 1.7.3 - Runtime_Tools Release Notes","doc":"","ref":"notes.html#runtime_tools-1-7-3"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Runtime_Tools Release Notes","doc":"- Fixed a timestamp problem where some events could be sent out of order. Minor\n  fixes to presentation of data.\n\n  Own Id: OTP-7544 Aux Id: otp-7442","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Runtime_Tools 1.7.2 - Runtime_Tools Release Notes","doc":"","ref":"notes.html#runtime_tools-1-7-2"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Runtime_Tools Release Notes","doc":"- `etop` now collects process information faster and more reliably than before\n  (a race condition reported by Igor Goryachev has been eliminated).\n\n  Trace messages could be lost when `ttb:stop/0` was called.\n\n  Own Id: OTP-7164","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Runtime_Tools 1.7.1 - Runtime_Tools Release Notes","doc":"","ref":"notes.html#runtime_tools-1-7-1"},{"type":"extras","title":"Improvements and New Features - Runtime_Tools Release Notes","doc":"- The documentation has been updated so as to reflect the last updates of the\n  Erlang shell as well as the minor modifications of the control sequence `p` of\n  the `io_lib` module.\n\n  Superfluous empty lines have been removed from code examples and from Erlang\n  shell examples.\n\n  Own Id: OTP-6944 Aux Id: OTP-6554, OTP-6911\n\n- Memory management improvements especially for the runtime system with SMP\n  support:\n\n  - The runtime system with SMP support can now use multiple, thread specific\n    instances of most memory allocators. This improves performance since it\n    reduces lock contention in the memory allocators. It may however increase\n    memory usage for some applications. The runtime system with SMP support will\n    by default enable this feature on most allocators. The amount of instances\n    used can be configured.\n  - `driver_alloc()`, `driver_realloc()`, and `driver_free()` now use their own\n    erts specific memory allocator instead of the default `malloc()`\n    implementation on the system.\n  - The default configuration of some allocators have been changed to fit\n    applications that use much memory better.\n  - Some new `erts_alloc` configuration parameters have been added.\n  - `erts_alloc_config` has been modified to be able to create configurations\n    suitable for multiple instances of allocators.\n  - The returned value from `erlang:system_info({allocator, Alloc})` has been\n    changed. This since an allocator may now run in multiple instances.\n\n  If you for some reason want the memory allocators to be configured as before,\n  you can pass the `+Mea r11b` command-line argument to `erl`.\n\n  For more information see the `erts_alloc(3)`, the `m:erts_alloc_config`, and\n  the `m:erlang` documentation.\n\n  \\*** POTENTIAL INCOMPATIBILITY \\***\n\n  Own Id: OTP-7100","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Runtime_Tools 1.7 - Runtime_Tools Release Notes","doc":"","ref":"notes.html#runtime_tools-1-7"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Runtime_Tools Release Notes","doc":"- `dbg` could leave traced processes in a suspended state if the tracer process\n  was killed with exit reason `kill`.\n\n  `erlang:suspend_process/2` has been introduced which accepts an option list as\n  second argument. For more information see the `m:erlang` documentation.\n\n  Processes suspended via `erlang:suspend_process/[1,2]` will now be\n  automatically resumed if the process that called\n  `erlang:suspend_process/[1,2]` terminates.\n\n  Processes could previously be suspended by one process and resumed by another\n  unless someone was tracing the suspendee. This is _not_ possible anymore. The\n  process resuming a process _has_ to be the one that suspended it.\n\n  \\*** POTENTIAL INCOMPATIBILITY \\***\n\n  Own Id: OTP-6946","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Runtime_Tools Release Notes","doc":"- The undocumented and unsupported function `dbg:tracer/1` has been removed. The\n  undocumented, unsupported, and broken function `dbg:i/1` has been removed.\n\n  Own Id: OTP-6939","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Runtime_Tools 1.6.8 - Runtime_Tools Release Notes","doc":"","ref":"notes.html#runtime_tools-1-6-8"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Runtime_Tools Release Notes","doc":"- In this release the following has been fixed and enhanced: Autostart: It is\n  now possible to configure modules that shall be loaded by the autostart\n  mechanism. This because it is not certain that all application systems make\n  use of the OTP boot script to set up paths to all Erlang modules.\n  Runtime_tools/Inviso: A bug in the fetch_log functionality has been fixed.\n  Further a bug that was (supposedly) fixed in a previous patch concerning\n  meta-tracer write_ti has been fixed (again) in this patch. A bug in\n  inviso_as_lib making remote autostart config file updates fail has been fixed.\n  Inviso: inviso_tool has been given a flush API.\n\n  Own Id: OTP-6918","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Runtime_Tools 1.6.7 - Runtime_Tools Release Notes","doc":"","ref":"notes.html#runtime_tools-1-6-7"},{"type":"extras","title":"Improvements and New Features - Runtime_Tools Release Notes","doc":"- The following bugs/improvements have been done: Internal interworking between\n  inviso_rt and inviso_rt_meta. The call function used by inviso_rt to call\n  inviso_rt_meta is now protected with a monitor. Inviso_rt_meta now includes\n  the timestamp of the incoming meta trace message when calling the\n  call-callback. (Makes it possible to add a \"better\" timestamp to the ti-file.)\n  Bug in inviso_tool making it not remove trace patterns when terminating. Bug\n  in internal function h_start_session making inviso_tool crash if there were no\n  active nodes to start the session on. The user-inviso_tool and inviso\n  API-inviso control component request/response gen_server protocols had default\n  time-out. Since many trace operations can be time consuming, a longer time-out\n  is necessary. Improved overload protection. It is now possible to let the\n  overload protection renew itself (e.g after an exit from an external overload\n  protector). Inviso_rt_meta now fully uses the exception_trace match spec\n  action term. Run Trace Case API (as in contrast to activate and deactivate\n  trace case APIs) in inviso_tool. Flush trace-port API added to inviso.\n  Get_session_data API added to inviso_tool. Improved inviso_tool:stop making it\n  possible to name nodes which shall not have their trace patterns removed when\n  inviso_tool terminates. Bug in handling of writing multiple ti-entries if\n  returned from a call/return_from call-back in inviso_rt_meta Process trace\n  flags are no longer explicitly removed by the inviso_tool when it terminates.\n  Not necessary. Inviso_tool get_autostart_data adopted to standard autostarter.\n\n  \\*** INCOMPATIBILITY with Meta trace call-backs are called with different\n  arguments now. \\***\n\n  Own Id: OTP-6881","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Runtime_Tools 1.6.6 - Runtime_Tools Release Notes","doc":"","ref":"notes.html#runtime_tools-1-6-6"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Runtime_Tools Release Notes","doc":"- A bug in inviso_rt_meta caused an ETS table containing information on\n  initiated (init_tpm) functions to be lost when suspending tracing. Further an\n  enhancement to inviso_rt has been introduced making it possible to activate\n  process trace flags based on globally registered names. It is then not an\n  error to activate a global name on a node where the name does not reside. The\n  process count in the return value will simply be set to zero (hence exactly\n  one node in the NodeResult part of the return value will indicate one matching\n  process found). A bug was found in fetch_log API. At the same time the\n  fetch_log functionality was enhanced to also offer flow control making fetcher\n  processes send chunks of transferred file data at a slower pace.\n\n  Own Id: OTP-6703","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Improvements and New Features - Runtime_Tools Release Notes","doc":"- Minor Makefile changes.\n\n  Own Id: OTP-6689 Aux Id: OTP-6742\n\n- An experimental tool called `erts_alloc_config` has been added.\n  `erts_alloc_config` is intended to be used to aid creation of an `erts_alloc`\n  configuration that is suitable for a limited number of runtime scenarios. For\n  more information see the `m:erts_alloc_config` documentation.\n\n  Own Id: OTP-6700","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Runtime_Tools 1.6.5 - Runtime_Tools Release Notes","doc":"","ref":"notes.html#runtime_tools-1-6-5"},{"type":"extras","title":"Improvements and New Features - Runtime_Tools Release Notes","doc":"- Misc improvements.\n\n  Own Id: OTP-6576","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Runtime_Tools 1.6.4 - Runtime_Tools Release Notes","doc":"","ref":"notes.html#runtime_tools-1-6-4"},{"type":"extras","title":"Improvements and New Features - Runtime_Tools Release Notes","doc":"- This application has been updated to eliminate warnings by Dialyzer.\n\n  Own Id: OTP-6551","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Runtime_Tools 1.6.3 - Runtime_Tools Release Notes","doc":"","ref":"notes.html#runtime_tools-1-6-3"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Runtime_Tools Release Notes","doc":"- This ticket includes several improvements and bugfixes to both runtime_tools\n  and inviso. The overload mechanism can now also react to incoming messages.\n  This is useful if an external overload watch-dog is used. Some improvements of\n  variable bindings has been done to the default autostart mechanism -\n  inviso_autostart_server. Autostart \"jobs\" can now be done in parallel,\n  allowing for some jobs to hang waiting for some parts of the traced system to\n  become ready before proceeding. Previously when using named meta-match-specs\n  (tpm_ms) ending up with zero match-specs still kept the meta trace pattern\n  active. This caused zero match-specs to be equal to unlimited meta tracing on\n  that particular function. If the internal database becomes empty of meta match\n  specs, meta trace pattern is removed for that function. Standard public loop\n  data in the inviso runtime meta tracer process is now extended to a 2-tuple.\n  The functions ctp/1 and ctpl/1 are added making it possible to remove trace\n  patterns for a list of functions rather than one by one. Inviso_rt_meta will\n  now accept a list of binaries to be output into the trace information file, in\n  additions to a single binary. Further it is also possible to make own output\n  to the trace information file using the write_ti/1 function. An error was\n  discovered in inviso_rt making the inviso_rt_meta remain rather than terminate\n  if the inviso_rt terminated due to \"running alone\" (not allowed to run without\n  a control component). A new tool, inviso_tool, has been added to the inviso\n  application.\n\n  Own Id: OTP-6426","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Runtime_Tools 1.6.2 - Runtime_Tools Release Notes","doc":"","ref":"notes.html#runtime_tools-1-6-2"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Runtime_Tools Release Notes","doc":"- Several minor bugs and race conditions eliminated in the runtime_tools and\n  observer applications.\n\n  Own Id: OTP-6265","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Runtime_Tools 1.6.1 - Runtime_Tools Release Notes","doc":"","ref":"notes.html#runtime_tools-1-6-1"},{"type":"extras","title":"Improvements and New Features - Runtime_Tools Release Notes","doc":"- There are new BIFs `erlang:spawn_monitor/1,3`, and the new option `monitor`\n  for `spawn_opt/2,3,4,5`.\n\n  The `observer_backend` module has been updated to handle the new BIFs.\n\n  Own Id: OTP-6281","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Runtime_Tools 1.6 - Runtime_Tools Release Notes","doc":"Added the runtime part of the Inviso tracer, see the new Inviso application for\nmore information. This also meant adding an application callback module and an\napplication supervisor tree for Runtime_Tools.","ref":"notes.html#runtime_tools-1-6"},{"type":"extras","title":"Runtime_Tools 1.5.1.1 - Runtime_Tools Release Notes","doc":"","ref":"notes.html#runtime_tools-1-5-1-1"},{"type":"extras","title":"Improvements and New Features - Runtime_Tools Release Notes","doc":"- The `dbg` manual page has been updated with information about how to avoid\n  deadlock when tracing.\n\n  Own Id: OTP-5373 Aux Id: seq9729","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Runtime_Tools 1.5.1 - Runtime_Tools Release Notes","doc":"","ref":"notes.html#runtime_tools-1-5-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Runtime_Tools Release Notes","doc":"- Linked in drivers in the Crypto, and Asn1 applications are now compiled with\n  the `-D_THREAD_SAFE` and `-D_REENTRANT` switches on unix when the emulator has\n  thread support enabled.\n\n  Linked in drivers on MacOSX are not compiled with the undocumented\n  `-lbundle1.o` switch anymore. Thanks to Sean Hinde who sent us a patch.\n\n  Linked in driver in Crypto, and port programs in SSL, now compiles on OSF1.\n\n  Minor `Makefile` improvements in Runtime_Tools.\n\n  Own Id: OTP-5346","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"LTTng and Erlang/OTP","doc":"<!--\n%CopyrightBegin%\n\nCopyright Ericsson AB 2023-2024. All Rights Reserved.\n\nLicensed under the Apache License, Version 2.0 (the \"License\");\nyou may not use this file except in compliance with the License.\nYou may obtain a copy of the License at\n\n    http://www.apache.org/licenses/LICENSE-2.0\n\nUnless required by applicable law or agreed to in writing, software\ndistributed under the License is distributed on an \"AS IS\" BASIS,\nWITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.\nSee the License for the specific language governing permissions and\nlimitations under the License.\n\n%CopyrightEnd%\n-->\n# LTTng and Erlang/OTP","ref":"lttng.html"},{"type":"extras","title":"Introduction - LTTng and Erlang/OTP","doc":"The Linux Trace Toolkit: next generation is an open source system software\npackage for correlated tracing of the Linux kernel, user applications and\nlibraries.\n\nFor more information, please visit [http://lttng.org](http://lttng.org)","ref":"lttng.html#introduction"},{"type":"extras","title":"Building Erlang/OTP with LTTng support - LTTng and Erlang/OTP","doc":"Configure and build Erlang with LTTng support:\n\nFor LTTng to work properly with Erlang/OTP you need the following packages\ninstalled:\n\n- LTTng-tools: a command line interface to control tracing sessions.\n- LTTng-UST: user space tracing library.\n\nOn Ubuntu this can be installed via `aptitude`:\n\n```text\n$ sudo aptitude install lttng-tools liblttng-ust-dev\n```\n\nSee [Installing LTTng](http://lttng.org/docs/#doc-installing-lttng) for more\ninformation on how to install LTTng on your system.\n\nAfter LTTng is properly installed on the system Erlang/OTP can be built with\nLTTng support.\n\n```text\n$ ./configure --with-dynamic-trace=lttng\n$ make\n```","ref":"lttng.html#building-erlang-otp-with-lttng-support"},{"type":"extras","title":"Dyntrace Tracepoints - LTTng and Erlang/OTP","doc":"All tracepoints are in the domain of `org_erlang_dyntrace`\n\nAll Erlang types are the string equivalent in LTTng.\n\n_process_spawn_\n\n- `pid : string` :: Process ID. Ex. `\"<0.131.0>\"`\n- `parent : string` :: Process ID. Ex. `\"<0.131.0>\"`\n- `entry : string` :: Code Location. Ex. `\"lists:sort/1\"`\n\nAvailable through `erlang:trace/3` with trace flag `procs` and\n`{tracer,dyntrace,[]}` as tracer module.\n\nExample:\n\n```erlang\nprocess_spawn: { cpu_id = 3 }, { pid = \"<0.131.0>\", parent = \"<0.130.0>\", entry = \"erlang:apply/2\" }\n```\n\n_process_link_\n\n- `to : string` :: Process ID or Port ID. Ex. `\"<0.131.0>\"`\n- `from : string` :: Process ID or Port ID. Ex. `\"<0.131.0>\"`\n- `type : string` :: `\"link\" | \"unlink\"`\n\nAvailable through `erlang:trace/3` with trace flag `procs` and\n`{tracer,dyntrace,[]}` as tracer module.\n\nExample:\n\n```text\nprocess_link: { cpu_id = 3 }, { from = \"<0.130.0>\", to = \"<0.131.0>\", type = \"link\" }\n```\n\n_process_exit_\n\n- `pid : string` :: Process ID. Ex. `\"<0.131.0>\"`\n- `reason : string` :: Exit reason. Ex. `\"normal\"`\n\nAvailable through `erlang:trace/3` with trace flag `procs` and\n`{tracer,dyntrace,[]}` as tracer module.\n\nExample:\n\n```text\nprocess_exit: { cpu_id = 3 }, { pid = \"<0.130.0>\", reason = \"normal\" }\n```\n\n_process_register_\n\n- `pid : string` :: Process ID. Ex. `\"<0.131.0>\"`\n- `name : string` :: Registered name. Ex. `\"logger\"`\n- `type : string` :: `\"register\" | \"unregister\"`\n\nExample:\n\n```erlang\nprocess_register: { cpu_id = 0 }, { pid = \"<0.128.0>\", name = \"dyntrace_lttng_SUITE\" type = \"register\" }\n```\n\n_process_scheduled_\n\n- `pid : string` :: Process ID. Ex. `\"<0.131.0>\"`\n- `entry : string` :: Code Location. Ex. `\"lists:sort/1\"`\n- `type : string` ::\n  `\"in\" | \"out\" | \"in_exiting\" | \"out_exiting\" | \"out_exited\"`\n\nAvailable through `erlang:trace/3` with trace flag `running` and\n`{tracer,dyntrace,[]}` as tracer module.\n\nExample:\n\n```erlang\nprocess_scheduled: { cpu_id = 0 }, { pid = \"<0.136.0>\", entry = \"erlang:apply/2\", type = \"in\" }\n```\n\n_port_open_\n\n- `pid : string` :: Process ID. Ex. `\"<0.131.0>\"`\n- `driver : string` :: Driver name. Ex. `\"tcp_inet\"`\n- `port : string` :: Port ID. Ex. `\"#Port<0.1031>\"`\n\nAvailable through `erlang:trace/3` with trace flag `ports` and\n`{tracer,dyntrace,[]}` as tracer module.\n\nExample:\n\n```text\nport_open: { cpu_id = 5 }, { pid = \"<0.131.0>\", driver = \"'/bin/sh -s unix:cmd'\", port = \"#Port<0.1887>\" }\n```\n\n_port_exit_\n\n- `port : string` :: Port ID. Ex. `\"#Port<0.1031>\"`\n- `reason : string` :: Exit reason. Ex. `\"normal\"`\n\nAvailable through `erlang:trace/3` with trace flag `ports` and\n`{tracer,dyntrace,[]}` as tracer module.\n\nExample:\n\n```text\nport_exit: { cpu_id = 5 }, { port = \"#Port<0.1887>\", reason = \"normal\" }\n```\n\n_port_link_\n\n- `to : string` :: Process ID. Ex. `\"<0.131.0>\"`\n- `from : string` :: Process ID. Ex. `\"<0.131.0>\"`\n- `type : string` :: `\"link\" | \"unlink\"`\n\nAvailable through `erlang:trace/3` with trace flag `ports` and\n`{tracer,dyntrace,[]}` as tracer module.\n\nExample:\n\n```erlang\nport_link: { cpu_id = 5 }, { from = \"#Port<0.1887>\", to = \"<0.131.0>\", type = \"unlink\" }\n```\n\n_port_scheduled_\n\nAvailable through `erlang:trace/3` with trace flag `running` and\n`{tracer,dyntrace,[]}` as tracer module.\n\n- `port : string` :: Port ID. Ex. `\"#Port<0.1031>\"`\n- `entry : string` :: Callback. Ex. `\"open\"`\n- `type : string` ::\n  `\"in\" | \"out\" | \"in_exiting\" | \"out_exiting\" | \"out_exited\"`\n\nExample:\n\n```erlang\nport_scheduled: { cpu_id = 5 }, { pid = \"#Port<0.1905>\", entry = \"close\", type = \"out\" }\n```\n\nAvailable through `erlang:trace/3` with trace flag `running` and\n`{tracer,dyntrace,[]}` as tracer module.\n\n_function_call_\n\n- `pid : string` :: Process ID. Ex. `\"<0.131.0>\"`\n- `entry : string` :: Code Location. Ex. `\"lists:sort/1\"`\n- `depth : integer` :: Stack depth. Ex. `0`\n\nAvailable through `erlang:trace/3` with trace flag `call` and\n`{tracer,dyntrace,[]}` as tracer module.\n\nExample:\n\n```erlang\nfunction_call: { cpu_id = 5 }, { pid = \"<0.145.0>\", entry = \"dyntrace_lttng_SUITE:'-t_call/1-fun-1-'/0\", depth = 0 }\n```\n\n_function_return_\n\n- `pid : string` :: Process ID. Ex. `\"<0.131.0>\"`\n- `entry : string` :: Code Location. Ex. `\"lists:sort/1\"`\n- `depth : integer` :: Stack depth. Ex. `0`\n\nAvailable through `erlang:trace/3` with trace flag `call` or `return_to` and\n`{tracer,dyntrace,[]}` as tracer module.\n\nExample:\n\n```erlang\nfunction_return: { cpu_id = 5 }, { pid = \"<0.145.0>\", entry = \"dyntrace_lttng_SUITE:waiter/0\", depth = 0 }\n```\n\n_function_exception_\n\n- `pid : string` :: Process ID. Ex. `\"<0.131.0>\"`\n- `entry : string` :: Code Location. Ex. `\"lists:sort/1\"`\n- `class : string` :: Error reason. Ex. `\"error\"`\n\nAvailable through `erlang:trace/3` with trace flag `call` and\n`{tracer,dyntrace,[]}` as tracer module.\n\nExample:\n\n```erlang\nfunction_exception: { cpu_id = 5 }, { pid = \"<0.144.0>\", entry = \"t:call_exc/1\", class = \"error\" }\n```\n\n_message_send_\n\n- `from : string` :: Process ID or Port ID. Ex. `\"<0.131.0>\"`\n- `to : string` :: Process ID or Port ID. Ex. `\"<0.131.0>\"`\n- `message : string` :: Message sent. Ex. `\"{<0.162.0>,ok}\"`\n\nAvailable through `erlang:trace/3` with trace flag `send` and\n`{tracer,dyntrace,[]}` as tracer module.\n\nExample:\n\n```text\nmessage_send: { cpu_id = 3 }, { from = \"#Port<0.1938>\", to = \"<0.160.0>\", message = \"{#Port<0.1938>,eof}\" }\n```\n\n_message_receive_\n\n- `to : string` :: Process ID or Port ID. Ex. `\"<0.131.0>\"`\n- `message : string` :: Message received. Ex. `\"{<0.162.0>,ok}\"`\n\nAvailable through `erlang:trace/3` with trace flag `'receive'` and\n`{tracer,dyntrace,[]}` as tracer module.\n\nExample:\n\n```text\nmessage_receive: { cpu_id = 7 }, { to = \"<0.167.0>\", message = \"{<0.165.0>,ok}\" }\n```\n\n_gc_minor_start_\n\n- `pid : string` :: Process ID. Ex. `\"<0.131.0>\"`\n- `need : integer` :: Heap need. Ex. `2`\n- `heap : integer` :: Young heap word size. Ex. `233`\n- `old_heap : integer` :: Old heap word size. Ex. `233`\n\nAvailable through `erlang:trace/3` with trace flag `garbage_collection` and\n`{tracer,dyntrace,[]}` as tracer module.\n\nExample:\n\n```erlang\ngc_minor_start: { cpu_id = 0 }, { pid = \"<0.172.0>\", need = 0, heap = 610, old_heap = 0 }\n```\n\n_gc_minor_end_\n\n- `pid : string` :: Process ID. Ex. `\"<0.131.0>\"`\n- `reclaimed : integer` :: Heap reclaimed. Ex. `2`\n- `heap : integer` :: Young heap word size. Ex. `233`\n- `old_heap : integer` :: Old heap word size. Ex. `233`\n\nAvailable through `erlang:trace/3` with trace flag `garbage_collection` and\n`{tracer,dyntrace,[]}` as tracer module.\n\nExample:\n\n```c\ngc_minor_end: { cpu_id = 0 }, { pid = \"<0.172.0>\", reclaimed = 120, heap = 1598, old_heap = 1598 }\n```\n\n_gc_major_start_\n\n- `pid : string` :: Process ID. Ex. `\"<0.131.0>\"`\n- `need : integer` :: Heap need. Ex. `2`\n- `heap : integer` :: Young heap word size. Ex. `233`\n- `old_heap : integer` :: Old heap word size. Ex. `233`\n\nAvailable through `erlang:trace/3` with trace flag `garbage_collection` and\n`{tracer,dyntrace,[]}` as tracer module.\n\nExample:\n\n```erlang\ngc_major_start: { cpu_id = 0 }, { pid = \"<0.172.0>\", need = 8, heap = 2586, old_heap = 1598 }\n```\n\n_gc_major_end_\n\n- `pid : string` :: Process ID. Ex. `\"<0.131.0>\"`\n- `reclaimed : integer` :: Heap reclaimed. Ex. `2`\n- `heap : integer` :: Young heap word size. Ex. `233`\n- `old_heap : integer` :: Old heap word size. Ex. `233`\n\nAvailable through `erlang:trace/3` with trace flag `garbage_collection` and\n`{tracer,dyntrace,[]}` as tracer module.\n\nExample:\n\n```text\ngc_major_end: { cpu_id = 0 }, { pid = \"<0.172.0>\", reclaimed = 240, heap = 4185, old_heap = 0 }\n```","ref":"lttng.html#dyntrace-tracepoints"},{"type":"extras","title":"BEAM Tracepoints - LTTng and Erlang/OTP","doc":"All tracepoints are in the domain of `org_erlang_otp`\n\nAll Erlang types are the string equivalent in LTTng.\n\n_driver_init_\n\n- `driver : string` :: Driver name. Ex. `\"tcp_inet\"`\n- `major : integer` :: Major version. Ex. `3`\n- `minor : integer` :: Minor version. Ex. `1`\n- `flags : integer` :: Flags. Ex. `1`\n\nExample:\n\n```erlang\ndriver_init: { cpu_id = 2 }, { driver = \"caller_drv\", major = 3, minor = 3, flags = 1 }\n```\n\n_driver_start_\n\n- `pid : string` :: Process ID. Ex. `\"<0.131.0>\"`\n- `driver : string` :: Driver name. Ex. `\"tcp_inet\"`\n- `port : string` :: Port ID. Ex. `\"#Port<0.1031>\"`\n\nExample:\n\n```erlang\ndriver_start: { cpu_id = 2 }, { pid = \"<0.198.0>\", driver = \"caller_drv\", port = \"#Port<0.3676>\" }\n```\n\n_driver_output_\n\n- `pid : string` :: Process ID. Ex. `\"<0.131.0>\"`\n- `port : string` :: Port ID. Ex. `\"#Port<0.1031>\"`\n- `driver : string` :: Driver name. Ex. `\"tcp_inet\"`\n- `bytes : integer` :: Size of data returned. Ex. `82`\n\nExample:\n\n```text\ndriver_output: { cpu_id = 2 }, { pid = \"<0.198.0>\", port = \"#Port<0.3677>\", driver = \"/bin/sh -s unix:cmd\", bytes = 36 }\n```\n\n_driver_outputv_\n\n- `pid : string` :: Process ID. Ex. `\"<0.131.0>\"`\n- `port : string` :: Port ID. Ex. `\"#Port<0.1031>\"`\n- `driver : string` :: Driver name. Ex. `\"tcp_inet\"`\n- `bytes : integer` :: Size of data returned. Ex. `82`\n\nExample:\n\n```erlang\ndriver_outputv: { cpu_id = 5 }, { pid = \"<0.194.0>\", port = \"#Port<0.3663>\", driver = \"tcp_inet\", bytes = 3 }\n```\n\n_driver_ready_input_\n\n- `pid : string` :: Process ID. Ex. `\"<0.131.0>\"`\n- `port : string` :: Port ID. Ex. `\"#Port<0.1031>\"`\n- `driver : string` :: Driver name. Ex. `\"tcp_inet\"`\n\nExample:\n\n```text\ndriver_ready_input: { cpu_id = 5 }, { pid = \"<0.189.0>\", port = \"#Port<0.3637>\", driver = \"inet_gethost 4 \" }\n```\n\n_driver_ready_output_\n\n- `pid : string` :: Process ID. Ex. `\"<0.131.0>\"`\n- `port : string` :: Port ID. Ex. `\"#Port<0.1031>\"`\n- `driver : string` :: Driver name. Ex. `\"tcp_inet\"`\n\nExample:\n\n```erlang\ndriver_ready_output: { cpu_id = 5 }, { pid = \"<0.194.0>\", port = \"#Port<0.3663>\", driver = \"tcp_inet\" }\n```\n\n_driver_timeout_\n\n- `pid : string` :: Process ID. Ex. `\"<0.131.0>\"`\n- `port : string` :: Port ID. Ex. `\"#Port<0.1031>\"`\n- `driver : string` :: Driver name. Ex. `\"tcp_inet\"`\n\nExample:\n\n```erlang\ndriver_timeout: { cpu_id = 5 }, { pid = \"<0.196.0>\", port = \"#Port<0.3664>\", driver = \"tcp_inet\" }\n```\n\n_driver_stop_select_\n\n- `driver : string` :: Driver name. Ex. `\"tcp_inet\"`\n\nExample:\n\n```erlang\ndriver_stop_select: { cpu_id = 5 }, { driver = \"unknown\" }\n```\n\n_driver_flush_\n\n- `pid : string` :: Process ID. Ex. `\"<0.131.0>\"`\n- `port : string` :: Port ID. Ex. `\"#Port<0.1031>\"`\n- `driver : string` :: Driver name. Ex. `\"tcp_inet\"`\n\nExample:\n\n```erlang\ndriver_flush: { cpu_id = 7 }, { pid = \"<0.204.0>\", port = \"#Port<0.3686>\", driver = \"tcp_inet\" }\n```\n\n_driver_stop_\n\n- `pid : string` :: Process ID. Ex. `\"<0.131.0>\"`\n- `port : string` :: Port ID. Ex. `\"#Port<0.1031>\"`\n- `driver : string` :: Driver name. Ex. `\"tcp_inet\"`\n\nExample:\n\n```erlang\ndriver_stop: { cpu_id = 5 }, { pid = \"[]\", port = \"#Port<0.3673>\", driver = \"tcp_inet\" }\n```\n\n_driver_process_exit_\n\n- `pid : string` :: Process ID. Ex. `\"<0.131.0>\"`\n- `port : string` :: Port ID. Ex. `\"#Port<0.1031>\"`\n- `driver : string` :: Driver name. Ex. `\"tcp_inet\"`\n\n_driver_ready_async_\n\n- `pid : string` :: Process ID. Ex. `\"<0.131.0>\"`\n- `port : string` :: Port ID. Ex. `\"#Port<0.1031>\"`\n- `driver : string` :: Driver name. Ex. `\"tcp_inet\"`\n\nExample:\n\n```erlang\ndriver_ready_async: { cpu_id = 3 }, { pid = \"<0.181.0>\", port = \"#Port<0.3622>\", driver = \"tcp_inet\" }\n```\n\n_driver_call_\n\n- `pid : string` :: Process ID. Ex. `\"<0.131.0>\"`\n- `port : string` :: Port ID. Ex. `\"#Port<0.1031>\"`\n- `driver : string` :: Driver name. Ex. `\"tcp_inet\"`\n- `command : integer` :: Command integer. Ex. `1`\n- `bytes : integer` :: Size of data returned. Ex. `82`\n\nExample:\n\n```erlang\ndriver_call: { cpu_id = 2 }, { pid = \"<0.202.0>\", port = \"#Port<0.3676>\", driver = \"caller_drv\", command = 0, bytes = 2 }\n```\n\n_driver_control_\n\n- `pid : string` :: Process ID. Ex. `\"<0.131.0>\"`\n- `port : string` :: Port ID. Ex. `\"#Port<0.1031>\"`\n- `driver : string` :: Driver name. Ex. `\"tcp_inet\"`\n- `command : integer` :: Command integer. Ex. `1`\n- `bytes : integer` :: Size of data returned. Ex. `82`\n\nExample:\n\n```erlang\ndriver_control: { cpu_id = 3 }, { pid = \"<0.32767.8191>\", port = \"#Port<0.0>\", driver = \"forker\", command = 83, bytes = 32 }\n```\n\n_carrier_create_\n\n- `type : string` :: Carrier type. Ex. `\"ets_alloc\"`\n- `instance : integer` :: Allocator instance. Ex. `1`\n- `size : integer` :: Carrier size. Ex. `262144`\n- `mbc_carriers : integer` :: Number of multiblock carriers in instance. Ex. `3`\n- `mbc_carriers_size : integer` :: Total size of multiblock blocks carriers in\n  instance. Ex. `1343488`\n- `mbc_blocks : integer` :: Number of multiblock blocks in instance. Ex. `122`\n- `mbc_blocks_size : integer` :: Total size of all multiblock blocks in\n  instance. Ex. `285296`\n- `sbc_carriers : integer` :: Number of singleblock carriers in instance. Ex.\n  `1`\n- `sbc_carriers_size : integer` :: Total size of singleblock blocks carriers in\n  instance. Ex. `1343488`\n- `sbc_blocks : integer` :: Number of singleblocks in instance. Ex. `1`\n- `sbc_blocks_size : integer` :: Total size of all singleblock blocks in\n  instance. Ex. `285296`\n\nExample:\n\n```c\ncarrier_create: { cpu_id = 2 }, { type = \"ets_alloc\", instance = 7, size = 2097152, mbc_carriers = 4, mbc_carriers_size = 3440640, mbc_blocks = 526, mbc_blocks_size = 1278576, sbc_carriers = 0, sbc_carriers_size = 0, sbc_blocks = 0, sbc_blocks_size = 0 }\n```\n\n_carrier_destroy_\n\n- `type : string` :: Carrier type. Ex. `\"ets_alloc\"`\n- `instance : integer` :: Allocator instance. Ex. `1`\n- `size : integer` :: Carrier size. Ex. `262144`\n- `mbc_carriers : integer` :: Number of multiblock carriers in instance. Ex. `3`\n- `mbc_carriers_size : integer` :: Total size of multiblock blocks carriers in\n  instance. Ex. `1343488`\n- `mbc_blocks : integer` :: Number of multiblock blocks in instance. Ex. `122`\n- `mbc_blocks_size : integer` :: Total size of all multiblock blocks in\n  instance. Ex. `285296`\n- `sbc_carriers : integer` :: Number of singleblock carriers in instance. Ex.\n  `1`\n- `sbc_carriers_size : integer` :: Total size of singleblock blocks carriers in\n  instance. Ex. `1343488`\n- `sbc_blocks : integer` :: Number of singleblocks in instance. Ex. `1`\n- `sbc_blocks_size : integer` :: Total size of all singleblock blocks in\n  instance. Ex. `285296`\n\nExample:\n\n```c\ncarrier_destroy: { cpu_id = 6 }, { type = \"ets_alloc\", instance = 7, size = 262144, mbc_carriers = 3, mbc_carriers_size = 3178496, mbc_blocks = 925, mbc_blocks_size = 2305336, sbc_carriers = 0, sbc_carriers_size = 0, sbc_blocks = 0, sbc_blocks_size = 0 }\n```\n\n_carrier_pool_put_\n\n- `type : string` :: Carrier type. Ex. `\"ets_alloc\"`\n- `instance : integer` :: Allocator instance. Ex. `1`\n- `size : integer` :: Carrier size. Ex. `262144`\n\nExample:\n\n```c\ncarrier_pool_put: { cpu_id = 3 }, { type = \"ets_alloc\", instance = 5, size = 1048576 }\n```\n\n_carrier_pool_get_\n\n- `type : string` :: Carrier type. Ex. `\"ets_alloc\"`\n- `instance : integer` :: Allocator instance. Ex. `1`\n- `size : integer` :: Carrier size. Ex. `262144`\n\nExample:\n\n```c\ncarrier_pool_get: { cpu_id = 7 }, { type = \"ets_alloc\", instance = 4, size = 3208 }\n```","ref":"lttng.html#beam-tracepoints"},{"type":"extras","title":"Example of process tracing - LTTng and Erlang/OTP","doc":"An example of process tracing of `os_mon` and friends.\n\nClean start of lttng in a bash shell.\n\n```text\n$ lttng create erlang-demo\nSpawning a session daemon\nSession erlang-demo created.\nTraces will be written in /home/egil/lttng-traces/erlang-demo-20160526-165920\n```\n\nStart an Erlang node with lttng enabled.\n\n```text\n$ erl\nErlang/OTP 19 [erts-8.0] [source-4d7b24d] [64-bit] [smp:8:8] [async-threads:10] [hipe] [kernel-poll:false] [lttng]\n\nEshell V8.0  (abort with ^G)\n1>\n```\n\nLoad the `dyntrace` module.\n\n```erlang\n1> l(dyntrace).\n{module,dyntrace}\n```\n\nAll tracepoints via dyntrace are now visible and can be listed through\n`lttng list -u`.\n\nEnable the process_register LTTng tracepoint for Erlang.\n\n```text\n$ lttng enable-event -u org_erlang_dyntrace:process_register\nUST event org_erlang_dyntrace:process_register created in channel channel0\n```\n\nEnable process tracing for new processes and use `dyntrace` as tracer backend.\n\n```erlang\n2> erlang:trace(new,true,[procs,{tracer,dyntrace,[]}]).\n0\n```\n\nStart LTTng tracing.\n\n```text\n$ lttng start\nTracing started for session erlang-demo\n```\n\nStart the `os_mon` application in Erlang.\n\n```text\n3> application:ensure_all_started(os_mon).\n{ok,[sasl,os_mon]}\n```\n\nStop LTTng tracing and view the result.\n\n```text\n$ lttng stop\nTracing stopped for session erlang-demo\n$ lttng view\n[17:20:42.561168759] (+?.?????????) elxd1168lx9 org_erlang_dyntrace:process_register: \\\n    { cpu_id = 5 }, { pid = \"<0.66.0>\", name = \"sasl_sup\", type = \"register\" }\n[17:20:42.561215519] (+0.000046760) elxd1168lx9 org_erlang_dyntrace:process_register: \\\n    { cpu_id = 5 }, { pid = \"<0.67.0>\", name = \"sasl_safe_sup\", type = \"register\" }\n[17:20:42.562149024] (+0.000933505) elxd1168lx9 org_erlang_dyntrace:process_register: \\\n    { cpu_id = 5 }, { pid = \"<0.68.0>\", name = \"alarm_handler\", type = \"register\" }\n[17:20:42.571035803] (+0.008886779) elxd1168lx9 org_erlang_dyntrace:process_register: \\\n    { cpu_id = 5 }, { pid = \"<0.69.0>\", name = \"release_handler\", type = \"register\" }\n[17:20:42.574939868] (+0.003904065) elxd1168lx9 org_erlang_dyntrace:process_register: \\\n    { cpu_id = 5 }, { pid = \"<0.74.0>\", name = \"os_mon_sup\", type = \"register\" }\n[17:20:42.576818712] (+0.001878844) elxd1168lx9 org_erlang_dyntrace:process_register: \\\n    { cpu_id = 5 }, { pid = \"<0.75.0>\", name = \"disksup\", type = \"register\" }\n[17:20:42.580032013] (+0.003213301) elxd1168lx9 org_erlang_dyntrace:process_register: \\\n    { cpu_id = 5 }, { pid = \"<0.76.0>\", name = \"memsup\", type = \"register\" }\n[17:20:42.583046339] (+0.003014326) elxd1168lx9 org_erlang_dyntrace:process_register: \\\n    { cpu_id = 5 }, { pid = \"<0.78.0>\", name = \"cpu_sup\", type = \"register\" }\n[17:20:42.586206242] (+0.003159903) elxd1168lx9 org_erlang_dyntrace:process_register: \\\n    { cpu_id = 5 }, { pid = \"<0.82.0>\", name = \"timer_server\", type = \"register\" }\n```","ref":"lttng.html#example-of-process-tracing"},{"type":"extras","title":"DTrace and Erlang/OTP","doc":"<!--\n%CopyrightBegin%\n\nCopyright Ericsson AB 2023. All Rights Reserved.\n\nLicensed under the Apache License, Version 2.0 (the \"License\");\nyou may not use this file except in compliance with the License.\nYou may obtain a copy of the License at\n\n    http://www.apache.org/licenses/LICENSE-2.0\n\nUnless required by applicable law or agreed to in writing, software\ndistributed under the License is distributed on an \"AS IS\" BASIS,\nWITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.\nSee the License for the specific language governing permissions and\nlimitations under the License.\n\n%CopyrightEnd%\n-->\n# DTrace and Erlang/OTP","ref":"dtrace.html"},{"type":"extras","title":"History - DTrace and Erlang/OTP","doc":"The first implementation of DTrace probes for the Erlang virtual machine was\npresented at the [2008 Erlang User Conference](https://erlang.org/euc/08/). That\nwork, based on the Erlang/OTP R12 release, was discontinued due to what appears\nto be miscommunication with the original developers.\n\nSeveral users have created Erlang port drivers, linked-in drivers, or NIFs that\nallow Erlang code to try to activate a probe, e.g.\n`foo_module:dtrace_probe(\"message goes here!\")`.","ref":"dtrace.html#history"},{"type":"extras","title":"Goals - DTrace and Erlang/OTP","doc":"- Annotate as much of the Erlang VM as is practical.\n- The initial goal is to trace file I/O operations.\n- Support all platforms that implement DTrace: OS X, Solaris, and (I hope)\n  FreeBSD and NetBSD.\n- To the extent that it's practical, support SystemTap on Linux via DTrace\n  provider compatibility.\n- Allow Erlang code to supply annotations.","ref":"dtrace.html#goals"},{"type":"extras","title":"Supported platforms - DTrace and Erlang/OTP","doc":"- OS X 10.6.x / Snow Leopard, OS X 10.7.x / Lion and probably newer versions.\n- Solaris 10. I have done limited testing on Solaris 11 and OpenIndiana release\n  151a, and both appear to work.\n- FreeBSD 9.0 and 10.0.\n- Linux via SystemTap compatibility. Please see\n  [$ERL_TOP/HOWTO/SYSTEMTAP.md](systemtap.md) for more details.\n\nJust add the `--with-dynamic-trace=dtrace` option to your command when you run\nthe `configure` script. If you are using systemtap, the configure option is\n`--with-dynamic-trace=systemtap`","ref":"dtrace.html#supported-platforms"},{"type":"extras","title":"Status - DTrace and Erlang/OTP","doc":"As of R15B01, the dynamic trace code is included in the OTP source distribution,\nalthough it's considered experimental. The main development of the dtrace code\nstill happens outside of Ericsson, but there is no need to fetch a patched\nversion of the OTP source to get the basic functionality.","ref":"dtrace.html#status"},{"type":"extras","title":"DTrace probe specifications - DTrace and Erlang/OTP","doc":"Probe specifications can be found in `erts/emulator/beam/erlang_dtrace.d`, and a\nfew example scripts can be found under `lib/runtime_tools/examples/`.","ref":"dtrace.html#dtrace-probe-specifications"},{"type":"extras","title":"SystemTap and Erlang/OTP","doc":"<!--\n%CopyrightBegin%\n\nCopyright Ericsson AB 2023. All Rights Reserved.\n\nLicensed under the Apache License, Version 2.0 (the \"License\");\nyou may not use this file except in compliance with the License.\nYou may obtain a copy of the License at\n\n    http://www.apache.org/licenses/LICENSE-2.0\n\nUnless required by applicable law or agreed to in writing, software\ndistributed under the License is distributed on an \"AS IS\" BASIS,\nWITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.\nSee the License for the specific language governing permissions and\nlimitations under the License.\n\n%CopyrightEnd%\n-->\n# SystemTap and Erlang/OTP","ref":"systemtap.html"},{"type":"extras","title":"Introduction - SystemTap and Erlang/OTP","doc":"SystemTap is DTrace for Linux. In fact Erlang's SystemTap support is built using\nSystemTap's DTrace compatibility's layer. For an introduction to Erlang DTrace\nsupport read [$ERL_TOP/HOWTO/DTRACE.md](dtrace.md).","ref":"systemtap.html#introduction"},{"type":"extras","title":"Requisites - SystemTap and Erlang/OTP","doc":"- Linux Kernel with UTRACE support\n\n  check for UTRACE support in your current kernel:\n\n  ```text\n  # grep CONFIG_UTRACE /boot/config-`uname -r`\n  CONFIG_UTRACE=y\n  ```\n\n  Fedora 16 is known to contain UTRACE, for most other Linux distributions a\n  custom build kernel will be required. Check Fedora's SystemTap documentation\n  for additional required packages (e.g. Kernel Debug Symbols)\n\n- SystemTap > 1.6\n\n  A the time of writing this, the latest released version of SystemTap is\n  version 1.6. Erlang's DTrace support requires a MACRO that was introduced\n  after that release. So either get a newer release or build SystemTap from git\n  yourself (see: http://sourceware.org/systemtap/getinvolved.html)","ref":"systemtap.html#requisites"},{"type":"extras","title":"Building Erlang - SystemTap and Erlang/OTP","doc":"Configure and build Erlang with SystemTap support:\n\n```text\n# ./configure --with-dynamic-trace=systemtap + whatever args you need\n# make\n```","ref":"systemtap.html#building-erlang"},{"type":"extras","title":"Testing - SystemTap and Erlang/OTP","doc":"SystemTap, unlike DTrace, needs to know what binary it is tracing and has to be\nable to read that binary before it starts tracing. Your probe script therefore\nhas to reference the correct beam emulator and stap needs to be able to find\nthat binary. The examples are written for \"beam\", but other versions such as\n\"beam.smp\" or \"beam.debug.smp\" might exist (depending on your configuration).\nMake sure you either specify the full the path of the binary in the probe or\nyour \"beam\" binary is in the search path.\n\nAll available probes can be listed like this:\n\n```text\n# stap -L 'process(\"beam\").mark(\"*\")'\n```\n\nor:\n\n```text\n# PATH=/path/to/beam:$PATH stap -L 'process(\"beam\").mark(\"*\")'\n```\n\nProbes in the dtrace.so NIF library like this:\n\n```text\n# PATH=/path/to/dtrace/priv/lib:$PATH stap -L 'process(\"dtrace.so\").mark(\"*\")'\n```","ref":"systemtap.html#testing"},{"type":"extras","title":"Running SystemTap scripts - SystemTap and Erlang/OTP","doc":"Adjust the process(\"beam\") reference to your beam version and attach the script\nto a running \"beam\" instance:\n\n```text\n# stap /path/to/probe/script/port1.systemtap -x  \n```","ref":"systemtap.html#running-systemtap-scripts"}],"content_type":"text/plain","producer":{"name":"ex_doc","version":[48,46,51,50,46,50]}}