"""Check if a host is in the Google Chrome HSTS Preload list"""

import functools
import os
import typing

__version__ = "2026.2.1"
__checksum__ = "2b50aa771a16d6d028f3d69261438e1756b32fd2259b7dc56870341e7e6c0cb7"
__all__ = ["in_hsts_preload"]

# fmt: off
_GTLD_INCLUDE_SUBDOMAINS = {b'amazon', b'android', b'app', b'audible', b'azure', b'bank', b'bing', b'boo', b'channel', b'chrome', b'dad', b'day', b'dev', b'eat', b'esq', b'fire', b'fly', b'foo', b'fujitsu', b'gle', b'gmail', b'google', b'hangout', b'hotmail', b'imdb', b'ing', b'insurance', b'kindle', b'meet', b'meme', b'microsoft', b'mov', b'new', b'nexus', b'office', b'page', b'phd', b'play', b'prime', b'prof', b'rsvp', b'search', b'silk', b'skype', b'windows', b'xbox', b'xn--cckwcxetd', b'xn--jlq480n2rg', b'youtube', b'zappos', b'zip'}  # noqa: E501
_JUMPTABLE = [[(0, 4), (4, 10), (14, 19), (33, 42), (75, 26), (101, 12), (113, 8), (121, 19), (140, 22), (162, 7), (169, 13), (182, 25), None, (207, 23), (230, 37), (267, 19), (286, 9), (295, 36), (331, 14), (345, 10), (355, 18), None, (373, 55), (428, 8), (436, 34), (470, 19), (489, 13), (502, 14), (516, 7), None, None, (523, 33), (556, 20), (576, 25), (601, 14), (615, 14), (629, 28), None, (657, 18), (675, 30), (705, 8), (713, 13), (726, 10), (736, 8), (744, 17), None, (761, 26), (787, 16), (803, 5), (808, 10), (818, 14), (832, 11), (843, 12), (855, 27), (882, 7), (889, 11), (900, 5), (905, 5), (910, 20), (930, 8), (938, 27), (965, 46), (1011, 25), (1036, 16), (1052, 8), (1060, 5), (1065, 22), (1087, 18), None, (1105, 32), (1137, 15), (1152, 24), (1176, 11), None, (1187, 10), (1197, 19), (1216, 18), (1234, 18), (1252, 5), (1257, 20), (1277, 26), (1303, 35), (1338, 19), (1357, 20), (1377, 46), (1423, 12), (1435, 14), (1449, 8), None, (1457, 10), (1467, 13), (1480, 20), (1500, 19), None, (1519, 13), (1532, 26), (1558, 11), (1569, 4), (1573, 22), (1595, 10), (1605, 15), (1620, 14), (1634, 12), (1646, 11), (1657, 21), (1678, 12), (1690, 25), None, (1715, 10), (1725, 14), (1739, 26), (1765, 45), (1810, 15), None, (1825, 11), (1836, 30), (1866, 21), (1887, 26), (1913, 6), (1919, 6), (1925, 20), (1945, 5), (1950, 38), (1988, 23), (2011, 6), (2017, 13), (2030, 8), (2038, 19), (2057, 12), (2069, 70), (2139, 55), None, (2194, 23), (2217, 16), (2233, 14), None, (2247, 24), (2271, 32), (2303, 6), (2309, 41), (2350, 22), (2372, 23), (2395, 23), (2418, 12), (2430, 8), (2438, 24), (2462, 12), (2474, 32), (2506, 25), (2531, 15), None, (2546, 46), (2592, 28), (2620, 13), (2633, 18), (2651, 26), (2677, 5), (2682, 42), (2724, 30), (2754, 16), (2770, 33), (2803, 17), (2820, 23), (2843, 14), (2857, 25), (2882, 19), (2901, 14), (2915, 7), (2922, 37), None, (2959, 18), (2977, 32), (3009, 27), (3036, 17), (3053, 24), (3077, 12), (3089, 28), (3117, 35), (3152, 12), (3164, 57), (3221, 32), (3253, 32), None, (3285, 8), (3293, 25), (3318, 18), (3336, 6), (3342, 23), None, (3365, 36), (3401, 33), (3434, 14), (3448, 16), (3464, 22), None, (3486, 20), (3506, 45), (3551, 34), (3585, 15), (3600, 20), (3620, 16), (3636, 21), (3657, 32), (3689, 24), (3713, 20), (3733, 10), (3743, 44), (3787, 6), (3793, 9), (3802, 12), (3814, 18), (3832, 5), (3837, 10), (3847, 39), (3886, 36), None, (3922, 25), (3947, 17), None, (3964, 14), (3978, 8), (3986, 7), None, (3993, 25), (4018, 17), None, (4035, 21), (4056, 35), (4091, 21), (4112, 10), (4122, 48), (4170, 11), (4181, 37), (4218, 46), (4264, 23), (4287, 12), (4299, 14), (4313, 23), (4336, 29), (4365, 14), (4379, 8), (4387, 47), (4434, 35), None, None, (4469, 47), (4516, 42), None, (4558, 18), None, (4576, 20), (4596, 8), (4604, 34), (4638, 6), (4644, 27), (4671, 22)], [(4693, 5347), (10040, 5033), (15073, 5617), (20690, 6001), (26691, 5965), (32656, 5192), (37848, 5768), (43616, 5446), (49062, 5183), (54245, 4849), (59094, 5240), (64334, 5387), (69721, 4948), (74669, 5799), (80468, 5484), (85952, 5496), (91448, 5152), (96600, 4897), (101497, 5388), (106885, 5095), (111980, 5159), (117139, 5319), (122458, 5826), (128284, 5100), (133384, 5518), (138902, 5001), (143903, 5361), (149264, 5407), (154671, 5338), (160009, 5113), (165122, 5183), (170305, 6043), (176348, 5265), (181613, 4954), (186567, 5044), (191611, 5888), (197499, 5436), (202935, 5761), (208696, 5824), (214520, 5810), (220330, 6325), (226655, 4880), (231535, 5431), (236966, 5394), (242360, 4769), (247129, 5531), (252660, 4903), (257563, 6099), (263662, 5091), (268753, 5183), (273936, 5820), (279756, 5239), (284995, 5253), (290248, 5564), (295812, 4977), (300789, 5020), (305809, 5916), (311725, 5694), (317419, 5279), (322698, 4987), (327685, 4825), (332510, 5570), (338080, 5509), (343589, 5640), (349229, 5910), (355139, 4573), (359712, 5492), (365204, 5941), (371145, 5727), (376872, 4965), (381837, 5263), (387100, 5524), (392624, 5043), (397667, 5439), (403106, 5289), (408395, 5994), (414389, 5176), (419565, 5783), (425348, 5494), (430842, 4952), (435794, 5124), (440918, 5201), (446119, 5004), (451123, 5283), (456406, 5574), (461980, 5580), (467560, 5703), (473263, 5083), (478346, 4954), (483300, 5476), (488776, 6006), (494782, 5701), (500483, 4855), (505338, 5102), (510440, 5361), (515801, 4863), (520664, 5733), (526397, 4736), (531133, 5791), (536924, 5113), (542037, 5379), (547416, 5255), (552671, 5517), (558188, 5287), (563475, 4699), (568174, 5379), (573553, 5635), (579188, 5185), (584373, 5164), (589537, 4489), (594026, 5487), (599513, 5246), (604759, 4694), (609453, 5471), (614924, 5463), (620387, 5109), (625496, 5561), (631057, 4954), (636011, 5692), (641703, 5527), (647230, 5743), (652973, 4943), (657916, 5658), (663574, 5489), (669063, 5396), (674459, 5919), (680378, 5398), (685776, 5303), (691079, 5193), (696272, 5500), (701772, 5941), (707713, 5376), (713089, 5280), (718369, 5084), (723453, 5166), (728619, 5842), (734461, 5567), (740028, 6014), (746042, 5494), (751536, 5511), (757047, 5959), (763006, 5584), (768590, 5637), (774227, 5801), (780028, 5699), (785727, 4957), (790684, 5550), (796234, 5524), (801758, 5187), (806945, 5225), (812170, 5159), (817329, 5451), (822780, 5037), (827817, 4986), (832803, 5474), (838277, 5334), (843611, 5288), (848899, 5265), (854164, 5412), (859576, 5044), (864620, 5412), (870032, 5329), (875361, 5352), (880713, 5286), (885999, 5073), (891072, 5803), (896875, 5671), (902546, 4896), (907442, 5758), (913200, 5147), (918347, 4892), (923239, 5432), (928671, 5174), (933845, 5570), (939415, 5183), (944598, 5160), (949758, 6216), (955974, 4753), (960727, 5081), (965808, 4835), (970643, 7039), (977682, 5507), (983189, 5478), (988667, 5469), (994136, 5608), (999744, 5101), (1004845, 4811), (1009656, 6080), (1015736, 5305), (1021041, 5444), (1026485, 5688), (1032173, 5006), (1037179, 5075), (1042254, 4847), (1047101, 5361), (1052462, 5520), (1057982, 5155), (1063137, 5296), (1068433, 4958), (1073391, 5437), (1078828, 5578), (1084406, 5922), (1090328, 5620), (1095948, 4936), (1100884, 5544), (1106428, 5857), (1112285, 5826), (1118111, 5638), (1123749, 6412), (1130161, 5031), (1135192, 6548), (1141740, 5297), (1147037, 5407), (1152444, 4819), (1157263, 5878), (1163141, 5469), (1168610, 5312), (1173922, 5861), (1179783, 4954), (1184737, 5625), (1190362, 5163), (1195525, 4768), (1200293, 5332), (1205625, 5295), (1210920, 5667), (1216587, 5199), (1221786, 6177), (1227963, 5425), (1233388, 5515), (1238903, 5776), (1244679, 5657), (1250336, 4976), (1255312, 5170), (1260482, 5490), (1265972, 4997), (1270969, 5675), (1276644, 5102), (1281746, 4983), (1286729, 4582), (1291311, 5631), (1296942, 4933), (1301875, 5304), (1307179, 5053), (1312232, 5743), (1317975, 4966), (1322941, 6188), (1329129, 5690), (1334819, 5250), (1340069, 5212), (1345281, 5067), (1350348, 5347), (1355695, 5461), (1361156, 4977), (1366133, 5266), (1371399, 5431), (1376830, 4938)], [(1381768, 738), (1382506, 470), (1382976, 734), (1383710, 673), (1384383, 619), (1385002, 675), (1385677, 516), (1386193, 516), (1386709, 582), (1387291, 595), (1387886, 473), (1388359, 546), (1388905, 681), (1389586, 764), (1390350, 581), (1390931, 726), (1391657, 1161), (1392818, 722), (1393540, 586), (1394126, 538), (1394664, 703), (1395367, 492), (1395859, 400), (1396259, 706), (1396965, 607), (1397572, 603), (1398175, 579), (1398754, 1138), (1399892, 622), (1400514, 587), (1401101, 471), (1401572, 670), (1402242, 551), (1402793, 554), (1403347, 700), (1404047, 683), (1404730, 481), (1405211, 634), (1405845, 458), (1406303, 593), (1406896, 599), (1407495, 530), (1408025, 448), (1408473, 603), (1409076, 544), (1409620, 556), (1410176, 586), (1410762, 533), (1411295, 428), (1411723, 514), (1412237, 593), (1412830, 477), (1413307, 806), (1414113, 810), (1414923, 1102), (1416025, 625), (1416650, 750), (1417400, 615), (1418015, 410), (1418425, 616), (1419041, 751), (1419792, 553), (1420345, 925), (1421270, 786), (1422056, 786), (1422842, 683), (1423525, 749), (1424274, 511), (1424785, 477), (1425262, 752), (1426014, 630), (1426644, 586), (1427230, 257), (1427487, 593), (1428080, 566), (1428646, 562), (1429208, 407), (1429615, 677), (1430292, 723), (1431015, 489), (1431504, 547), (1432051, 599), (1432650, 703), (1433353, 633), (1433986, 624), (1434610, 696), (1435306, 615), (1435921, 511), (1436432, 692), (1437124, 588), (1437712, 918), (1438630, 826), (1439456, 538), (1439994, 536), (1440530, 607), (1441137, 393), (1441530, 704), (1442234, 515), (1442749, 327), (1443076, 675), (1443751, 553), (1444304, 555), (1444859, 437), (1445296, 540), (1445836, 472), (1446308, 469), (1446777, 580), (1447357, 410), (1447767, 501), (1448268, 583), (1448851, 603), (1449454, 904), (1450358, 870), (1451228, 664), (1451892, 573), (1452465, 472), (1452937, 559), (1453496, 750), (1454246, 513), (1454759, 527), (1455286, 822), (1456108, 547), (1456655, 640), (1457295, 850), (1458145, 617), (1458762, 767), (1459529, 545), (1460074, 558), (1460632, 522), (1461154, 760), (1461914, 651), (1462565, 506), (1463071, 553), (1463624, 594), (1464218, 403), (1464621, 604), (1465225, 732), (1465957, 537), (1466494, 635), (1467129, 735), (1467864, 623), (1468487, 454), (1468941, 582), (1469523, 379), (1469902, 689), (1470591, 624), (1471215, 722), (1471937, 366), (1472303, 642), (1472945, 711), (1473656, 509), (1474165, 436), (1474601, 665), (1475266, 608), (1475874, 710), (1476584, 551), (1477135, 372), (1477507, 724), (1478231, 683), (1478914, 686), (1479600, 740), (1480340, 495), (1480835, 719), (1481554, 618), (1482172, 760), (1482932, 730), (1483662, 588), (1484250, 673), (1484923, 528), (1485451, 540), (1485991, 572), (1486563, 442), (1487005, 1001), (1488006, 678), (1488684, 843), (1489527, 666), (1490193, 556), (1490749, 614), (1491363, 696), (1492059, 550), (1492609, 1642), (1494251, 570), (1494821, 586), (1495407, 616), (1496023, 738), (1496761, 494), (1497255, 495), (1497750, 483), (1498233, 611), (1498844, 811), (1499655, 451), (1500106, 423), (1500529, 422), (1500951, 893), (1501844, 492), (1502336, 675), (1503011, 730), (1503741, 685), (1504426, 656), (1505082, 649), (1505731, 582), (1506313, 765), (1507078, 558), (1507636, 621), (1508257, 602), (1508859, 677), (1509536, 770), (1510306, 758), (1511064, 619), (1511683, 670), (1512353, 621), (1512974, 645), (1513619, 675), (1514294, 973), (1515267, 537), (1515804, 638), (1516442, 601), (1517043, 457), (1517500, 400), (1517900, 272), (1518172, 468), (1518640, 569), (1519209, 556), (1519765, 659), (1520424, 442), (1520866, 495), (1521361, 562), (1521923, 669), (1522592, 643), (1523235, 529), (1523764, 770), (1524534, 531), (1525065, 822), (1525887, 585), (1526472, 558), (1527030, 533), (1527563, 774), (1528337, 666), (1529003, 731), (1529734, 683), (1530417, 562), (1530979, 547), (1531526, 658), (1532184, 529), (1532713, 611), (1533324, 409), (1533733, 503), (1534236, 791), (1535027, 524), (1535551, 592), (1536143, 1958), (1538101, 453), (1538554, 781), (1539335, 577), (1539912, 590), (1540502, 747)], [(1541249, 48), None, None, (1541297, 42), None, (1541339, 27), (1541366, 25), None, None, None, None, None, None, None, None, None, None, (1541391, 42), (1541433, 20), None, (1541453, 44), None, None, (1541497, 18), None, (1541515, 23), None, None, None, None, None, None, (1541538, 21), (1541559, 25), None, None, (1541584, 26), None, None, None, None, (1541610, 44), (1541654, 21), (1541675, 23), None, None, None, None, None, None, None, None, None, None, (1541698, 31), None, None, None, None, (1541729, 42), (1541771, 19), (1541790, 16), None, (1541806, 21), None, None, (1541827, 42), None, None, (1541869, 42), (1541911, 27), None, None, None, None, (1541938, 37), (1541975, 21), None, None, (1541996, 20), (1542016, 42), None, None, None, (1542058, 25), (1542083, 17), None, (1542100, 21), None, None, None, (1542121, 24), None, (1542145, 24), (1542169, 21), None, None, (1542190, 22), None, None, (1542212, 17), (1542229, 19), None, None, None, None, (1542248, 31), None, None, None, None, None, None, None, None, (1542279, 42), (1542321, 42), (1542363, 17), (1542380, 17), None, None, (1542397, 19), None, (1542416, 25), None, None, (1542441, 20), None, None, (1542461, 42), (1542503, 63), None, None, None, (1542566, 21), None, None, None, None, (1542587, 21), (1542608, 16), (1542624, 17), (1542641, 31), None, None, None, None, (1542672, 42), None, (1542714, 25), None, (1542739, 9), None, (1542748, 21), (1542769, 42), None, None, (1542811, 65), (1542876, 56), (1542932, 21), None, (1542953, 42), None, None, (1542995, 24), None, None, None, None, None, None, (1543019, 42), (1543061, 21), (1543082, 21), None, (1543103, 42), (1543145, 25), None, (1543170, 37), (1543207, 21), (1543228, 42), None, None, (1543270, 21), (1543291, 16), None, None, (1543307, 16), None, (1543323, 51), None, None, (1543374, 21), None, (1543395, 22), (1543417, 21), (1543438, 21), None, None, (1543459, 63), (1543522, 41), (1543563, 39), (1543602, 42), None, None, None, None, None, None, (1543644, 21), (1543665, 21), None, None, (1543686, 21), None, None, (1543707, 21), None, None, None, (1543728, 50), None, None, None, (1543778, 50), None, (1543828, 21), (1543849, 21), (1543870, 19), None, (1543889, 16), (1543905, 26), None, (1543931, 58), (1543989, 42), None, None, None, None, None, None, (1544031, 41), None, None, None, (1544072, 21), None, None, (1544093, 44), None, (1544137, 254), (1544391, 21), None, None, (1544412, 21), None], [None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None]]  # noqa: E501
_CRC8_TABLE = [
    0x00, 0x07, 0x0e, 0x09, 0x1c, 0x1b, 0x12, 0x15,
    0x38, 0x3f, 0x36, 0x31, 0x24, 0x23, 0x2a, 0x2d,
    0x70, 0x77, 0x7e, 0x79, 0x6c, 0x6b, 0x62, 0x65,
    0x48, 0x4f, 0x46, 0x41, 0x54, 0x53, 0x5a, 0x5d,
    0xe0, 0xe7, 0xee, 0xe9, 0xfc, 0xfb, 0xf2, 0xf5,
    0xd8, 0xdf, 0xd6, 0xd1, 0xc4, 0xc3, 0xca, 0xcd,
    0x90, 0x97, 0x9e, 0x99, 0x8c, 0x8b, 0x82, 0x85,
    0xa8, 0xaf, 0xa6, 0xa1, 0xb4, 0xb3, 0xba, 0xbd,
    0xc7, 0xc0, 0xc9, 0xce, 0xdb, 0xdc, 0xd5, 0xd2,
    0xff, 0xf8, 0xf1, 0xf6, 0xe3, 0xe4, 0xed, 0xea,
    0xb7, 0xb0, 0xb9, 0xbe, 0xab, 0xac, 0xa5, 0xa2,
    0x8f, 0x88, 0x81, 0x86, 0x93, 0x94, 0x9d, 0x9a,
    0x27, 0x20, 0x29, 0x2e, 0x3b, 0x3c, 0x35, 0x32,
    0x1f, 0x18, 0x11, 0x16, 0x03, 0x04, 0x0d, 0x0a,
    0x57, 0x50, 0x59, 0x5e, 0x4b, 0x4c, 0x45, 0x42,
    0x6f, 0x68, 0x61, 0x66, 0x73, 0x74, 0x7d, 0x7a,
    0x89, 0x8e, 0x87, 0x80, 0x95, 0x92, 0x9b, 0x9c,
    0xb1, 0xb6, 0xbf, 0xb8, 0xad, 0xaa, 0xa3, 0xa4,
    0xf9, 0xfe, 0xf7, 0xf0, 0xe5, 0xe2, 0xeb, 0xec,
    0xc1, 0xc6, 0xcf, 0xc8, 0xdd, 0xda, 0xd3, 0xd4,
    0x69, 0x6e, 0x67, 0x60, 0x75, 0x72, 0x7b, 0x7c,
    0x51, 0x56, 0x5f, 0x58, 0x4d, 0x4a, 0x43, 0x44,
    0x19, 0x1e, 0x17, 0x10, 0x05, 0x02, 0x0b, 0x0c,
    0x21, 0x26, 0x2f, 0x28, 0x3d, 0x3a, 0x33, 0x34,
    0x4e, 0x49, 0x40, 0x47, 0x52, 0x55, 0x5c, 0x5b,
    0x76, 0x71, 0x78, 0x7f, 0x6a, 0x6d, 0x64, 0x63,
    0x3e, 0x39, 0x30, 0x37, 0x22, 0x25, 0x2c, 0x2b,
    0x06, 0x01, 0x08, 0x0f, 0x1a, 0x1d, 0x14, 0x13,
    0xae, 0xa9, 0xa0, 0xa7, 0xb2, 0xb5, 0xbc, 0xbb,
    0x96, 0x91, 0x98, 0x9f, 0x8a, 0x8d, 0x84, 0x83,
    0xde, 0xd9, 0xd0, 0xd7, 0xc2, 0xc5, 0xcc, 0xcb,
    0xe6, 0xe1, 0xe8, 0xef, 0xfa, 0xfd, 0xf4, 0xf3
]
# fmt: on

_IS_LEAF = 0x80
_INCLUDE_SUBDOMAINS = 0x40


try:
    from importlib.resources import open_binary

    def open_pkg_binary(path: str) -> typing.BinaryIO:
        return open_binary("hstspreload", path)

except ImportError:

    def open_pkg_binary(path: str) -> typing.BinaryIO:
        return open(
            os.path.join(os.path.dirname(os.path.abspath(__file__)), path),
            "rb",
        )


@functools.lru_cache(maxsize=1024)
def in_hsts_preload(host: typing.AnyStr) -> bool:
    """Determines if an IDNA-encoded host is on the HSTS preload list"""

    if isinstance(host, str):
        host = host.encode("ascii")
    labels = host.lower().split(b".")

    # Fast-branch for gTLDs that are registered to preload all sub-domains.
    if labels[-1] in _GTLD_INCLUDE_SUBDOMAINS:
        return True

    with open_pkg_binary("hstspreload.bin") as f:
        for layer, label in enumerate(labels[::-1]):
            # None of our layers are greater than 5 deep.
            if layer > 4:
                return False

            # Read the jump table for the layer and label
            jump_info = _JUMPTABLE[layer][_crc8(label)]
            if jump_info is None:
                # No entry: host is not preloaded
                return False

            # Read the set of entries for that layer and label
            f.seek(jump_info[0])
            data = bytearray(jump_info[1])
            f.readinto(data)

            for is_leaf, include_subdomains, ent_label in _iter_entries(data):
                # We found a potential leaf
                if is_leaf:
                    if ent_label == host:
                        return True
                    if include_subdomains and host.endswith(b"." + ent_label):
                        return True

                # Continue traversing as we're not at a leaf.
                elif label == ent_label:
                    break
            else:
                return False
    return False


def _iter_entries(data: bytes) -> typing.Iterable[typing.Tuple[int, int, bytes]]:
    while data:
        flags = data[0]
        size = data[1]
        label = bytes(data[2 : 2 + size])
        yield (flags & _IS_LEAF, flags & _INCLUDE_SUBDOMAINS, label)
        data = data[2 + size :]


def _crc8(value: bytes) -> int:
    # CRC8 reference implementation: https://github.com/niccokunzmann/crc8
    checksum = 0x00
    for byte in value:
        checksum = _CRC8_TABLE[checksum ^ byte]
    return checksum
