"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = __importDefault(require("assert"));
const path_1 = __importDefault(require("path"));
const sinon_1 = __importDefault(require("sinon"));
const typescript_1 = __importDefault(require("typescript"));
const documents_1 = require("../../../src/lib/documents");
const SnapshotManager_1 = require("../../../src/plugins/typescript/SnapshotManager");
const service_1 = require("../../../src/plugins/typescript/service");
const utils_1 = require("../../../src/utils");
const test_utils_1 = require("./test-utils");
describe('service', () => {
    const testDir = path_1.default.join(__dirname, 'testfiles');
    function setup() {
        const virtualSystem = (0, test_utils_1.createVirtualTsSystem)(testDir);
        const rootUris = [(0, utils_1.pathToUrl)(testDir)];
        const lsDocumentContext = {
            isSvelteCheck: false,
            ambientTypesSource: 'svelte2tsx',
            createDocument(fileName, content) {
                return new documents_1.Document((0, utils_1.pathToUrl)(fileName), content);
            },
            extendedConfigCache: new Map(),
            globalSnapshotsManager: new SnapshotManager_1.GlobalSnapshotsManager(virtualSystem),
            transformOnTemplateError: true,
            tsSystem: virtualSystem,
            watchTsConfig: false,
            notifyExceedSizeLimit: undefined,
            onProjectReloaded: undefined,
            projectService: undefined,
            nonRecursiveWatchPattern: undefined,
            watchDirectory: undefined,
            reportConfigError: undefined
        };
        return { virtualSystem, lsDocumentContext, rootUris };
    }
    it('can find tsconfig and override with default config', async () => {
        const dirPath = (0, test_utils_1.getRandomVirtualDirPath)(testDir);
        const { virtualSystem, lsDocumentContext, rootUris } = setup();
        virtualSystem.writeFile(path_1.default.join(dirPath, 'tsconfig.json'), JSON.stringify({
            compilerOptions: {
                checkJs: true,
                strict: true
            }
        }));
        virtualSystem.writeFile(path_1.default.join(dirPath, 'random.svelte'), '<script>const a: number = null;</script>');
        const ls = await (0, service_1.getService)(path_1.default.join(dirPath, 'random.svelte'), rootUris, lsDocumentContext);
        // ts internal
        delete ls.compilerOptions.configFilePath;
        assert_1.default.deepStrictEqual(ls.compilerOptions, {
            allowNonTsExtensions: true,
            checkJs: true,
            strict: true,
            module: typescript_1.default.ModuleKind.ESNext,
            moduleResolution: typescript_1.default.ModuleResolutionKind.Node10,
            target: typescript_1.default.ScriptTarget.ESNext
        });
    });
    it('errors if tsconfig matches no svelte files', async () => {
        const dirPath = (0, test_utils_1.getRandomVirtualDirPath)(testDir);
        const { virtualSystem, lsDocumentContext, rootUris } = setup();
        virtualSystem.readDirectory = () => [path_1.default.join(dirPath, 'random.ts')];
        virtualSystem.writeFile(path_1.default.join(dirPath, 'tsconfig.json'), JSON.stringify({
            include: ['**/*.ts']
        }));
        virtualSystem.writeFile(path_1.default.join(dirPath, 'random.svelte'), '<script>const a: number = null;</script>');
        let called = false;
        await (0, service_1.getService)(path_1.default.join(dirPath, 'random.svelte'), rootUris, {
            ...lsDocumentContext,
            reportConfigError: (message) => {
                called = true;
                assert_1.default.equal(message.uri, (0, utils_1.pathToUrl)(path_1.default.join(dirPath, 'tsconfig.json')));
            }
        });
        assert_1.default.ok(called);
    });
    it('does not report no svelte files when loaded through import', async () => {
        const dirPath = (0, test_utils_1.getRandomVirtualDirPath)(testDir);
        const { virtualSystem, lsDocumentContext, rootUris } = setup();
        virtualSystem.readDirectory = () => [path_1.default.join(dirPath, 'random.ts')];
        virtualSystem.writeFile(path_1.default.join(dirPath, 'tsconfig.json'), JSON.stringify({
            include: ['**/*.ts']
        }));
        virtualSystem.writeFile(path_1.default.join(dirPath, 'random.svelte'), '<script lang="ts">const a: number = null;</script>');
        virtualSystem.writeFile(path_1.default.join(dirPath, 'random.ts'), 'import {} from "./random.svelte"');
        let called = false;
        const service = await (0, service_1.getService)(path_1.default.join(dirPath, 'random.svelte'), rootUris, {
            ...lsDocumentContext,
            reportConfigError: (message) => {
                called = true;
                assert_1.default.deepStrictEqual([], message.diagnostics);
            }
        });
        assert_1.default.equal((0, utils_1.normalizePath)(path_1.default.join(dirPath, 'tsconfig.json')), (0, utils_1.normalizePath)(service.tsconfigPath));
        assert_1.default.ok(called);
    });
    it('does not errors if referenced tsconfig matches no svelte files', async () => {
        const dirPath = (0, test_utils_1.getRandomVirtualDirPath)(testDir);
        const { virtualSystem, lsDocumentContext, rootUris } = setup();
        const tsPattern = '**/*.ts';
        const sveltePattern = '**/*.svelte';
        virtualSystem.readDirectory = (_path, _extensions, _excludes, include) => {
            return include?.[0] === tsPattern
                ? [path_1.default.join(dirPath, 'random.ts')]
                : include?.[0] === sveltePattern
                    ? [path_1.default.join(dirPath, 'random.svelte')]
                    : [];
        };
        virtualSystem.writeFile(path_1.default.join(dirPath, 'tsconfig.json'), JSON.stringify({
            include: [],
            references: [{ path: './tsconfig_node.json' }, { path: './tsconfig_web.json' }]
        }));
        virtualSystem.writeFile(path_1.default.join(dirPath, 'tsconfig_node.json'), JSON.stringify({
            include: [tsPattern]
        }));
        virtualSystem.writeFile(path_1.default.join(dirPath, 'tsconfig_web.json'), JSON.stringify({
            include: [sveltePattern]
        }));
        virtualSystem.writeFile(path_1.default.join(dirPath, 'random.svelte'), '<script>const a: number = null;</script>');
        let called = false;
        const lsContainer = await (0, service_1.getService)(path_1.default.join(dirPath, 'random.svelte'), rootUris, {
            ...lsDocumentContext,
            reportConfigError: () => {
                called = true;
            }
        });
        assert_1.default.equal((0, utils_1.normalizePath)(path_1.default.join(dirPath, 'tsconfig_web.json')), lsContainer.tsconfigPath);
        assert_1.default.equal(called, false, 'expected not to call reportConfigError');
    });
    it('can loads default tsconfig', async () => {
        const dirPath = (0, test_utils_1.getRandomVirtualDirPath)(testDir);
        const { lsDocumentContext, rootUris } = setup();
        const ls = await (0, service_1.getService)(path_1.default.join(dirPath, 'random.svelte'), rootUris, lsDocumentContext);
        assert_1.default.deepStrictEqual(ls.compilerOptions, {
            allowJs: true,
            allowSyntheticDefaultImports: true,
            allowNonTsExtensions: true,
            configFilePath: undefined,
            declaration: false,
            maxNodeModuleJsDepth: 2,
            module: typescript_1.default.ModuleKind.ESNext,
            moduleResolution: typescript_1.default.ModuleResolutionKind.Node10,
            noEmit: true,
            skipLibCheck: true,
            target: typescript_1.default.ScriptTarget.ESNext
        });
    });
    it('patch release document so dispose do not throw', async () => {
        // testing this because the patch rely on ts implementation details
        // and we want to be aware of the changes
        const dirPath = (0, test_utils_1.getRandomVirtualDirPath)(testDir);
        const { virtualSystem, lsDocumentContext, rootUris } = setup();
        virtualSystem.writeFile(path_1.default.join(dirPath, 'tsconfig.json'), JSON.stringify({
            compilerOptions: {
                module: 'NodeNext',
                moduleResolution: 'NodeNext'
            }
        }));
        virtualSystem.writeFile(path_1.default.join(dirPath, 'random.svelte'), '<script>const a: number = null;</script>');
        const ls = await (0, service_1.getService)(path_1.default.join(dirPath, 'random.svelte'), rootUris, lsDocumentContext);
        const document = new documents_1.Document((0, utils_1.pathToUrl)(path_1.default.join(dirPath, 'random.svelte')), '');
        document.openedByClient = true;
        ls.updateSnapshot(document);
        const document2 = new documents_1.Document((0, utils_1.pathToUrl)(path_1.default.join(dirPath, 'random2.svelte')), '<script>import Random from "./random.svelte";</script>');
        document.openedByClient = true;
        ls.updateSnapshot(document2);
        const lang = ls.getService();
        lang.getProgram();
        // ensure updated document also works
        document2.update([
            {
                text: ' ',
                range: { start: { line: 0, character: 0 }, end: { line: 0, character: 0 } }
            }
        ]);
        lang.getProgram();
        assert_1.default.doesNotThrow(() => {
            lang.dispose();
        });
    });
    it('do not throw when script tag is nuked', async () => {
        // testing this because the patch rely on ts implementation details
        // and we want to be aware of the changes
        const dirPath = (0, test_utils_1.getRandomVirtualDirPath)(testDir);
        const { virtualSystem, lsDocumentContext, rootUris } = setup();
        virtualSystem.writeFile(path_1.default.join(dirPath, 'tsconfig.json'), JSON.stringify({
            compilerOptions: {
                module: 'NodeNext',
                moduleResolution: 'NodeNext'
            }
        }));
        virtualSystem.writeFile(path_1.default.join(dirPath, 'random.svelte'), '<script>const a: number = null;</script>');
        virtualSystem.writeFile(path_1.default.join(dirPath, 'random2.svelte'), '<script lang="ts">import Random from "./random.svelte";</script>');
        const ls = await (0, service_1.getService)(path_1.default.join(dirPath, 'random.svelte'), rootUris, lsDocumentContext);
        const document = new documents_1.Document((0, utils_1.pathToUrl)(path_1.default.join(dirPath, 'random.svelte')), '');
        document.openedByClient = true;
        ls.updateSnapshot(document);
        const document2 = new documents_1.Document((0, utils_1.pathToUrl)(path_1.default.join(dirPath, 'random2.svelte')), virtualSystem.readFile(path_1.default.join(dirPath, 'random2.svelte')));
        document.openedByClient = true;
        ls.updateSnapshot(document2);
        const lang = ls.getService();
        lang.getProgram();
        document2.update([{ text: '<script' }]);
        ls.updateSnapshot(document2);
        ls.getService();
    });
    function createReloadTester(docContext, testAfterReload) {
        let _resolve;
        let _reject;
        const reloadPromise = new Promise((resolve, reject) => {
            _resolve = resolve;
            _reject = reject;
        });
        return {
            docContextWithReload: {
                ...docContext,
                async onProjectReloaded(reloadingConfigs) {
                    try {
                        await testAfterReload(reloadingConfigs);
                        _resolve();
                    }
                    catch (e) {
                        _reject(e);
                    }
                }
            },
            reloadPromise
        };
    }
    it('can watch tsconfig', async () => {
        const dirPath = (0, test_utils_1.getRandomVirtualDirPath)(testDir);
        const { virtualSystem, lsDocumentContext, rootUris } = setup();
        const tsconfigPath = path_1.default.join(dirPath, 'tsconfig.json');
        virtualSystem.writeFile(tsconfigPath, JSON.stringify({
            compilerOptions: {
                strict: false
            }
        }));
        virtualSystem.writeFile(path_1.default.join(dirPath, 'random.svelte'), '<script>const a: number = null;</script>');
        const { reloadPromise, docContextWithReload } = createReloadTester({ ...lsDocumentContext, watchTsConfig: true }, testAfterReload);
        await (0, service_1.getService)(path_1.default.join(dirPath, 'random.svelte'), rootUris, docContextWithReload);
        virtualSystem.writeFile(tsconfigPath, JSON.stringify({
            compilerOptions: {
                strict: true
            }
        }));
        await reloadPromise;
        async function testAfterReload() {
            const newLs = await (0, service_1.getService)(path_1.default.join(dirPath, 'random.svelte'), rootUris, {
                ...lsDocumentContext,
                watchTsConfig: true
            });
            assert_1.default.strictEqual(newLs.compilerOptions.strict, true, 'expected to reload compilerOptions');
            return true;
        }
    });
    it('can watch extended tsconfig', async () => {
        const dirPath = (0, test_utils_1.getRandomVirtualDirPath)(testDir);
        const { virtualSystem, lsDocumentContext, rootUris } = setup();
        const tsconfigPath = path_1.default.join(dirPath, 'tsconfig.json');
        const extend = './.svelte-kit/tsconfig.json';
        const extendedConfigPathFull = path_1.default.resolve(path_1.default.dirname(tsconfigPath), extend);
        virtualSystem.writeFile(tsconfigPath, JSON.stringify({
            extends: extend
        }));
        virtualSystem.writeFile(path_1.default.join(dirPath, 'random.svelte'), '<script>const a: number = null;</script>');
        const { reloadPromise, docContextWithReload } = createReloadTester({ ...lsDocumentContext, watchTsConfig: true }, testAfterReload);
        await (0, service_1.getService)(path_1.default.join(dirPath, 'random.svelte'), rootUris, docContextWithReload);
        virtualSystem.writeFile(extendedConfigPathFull, JSON.stringify({
            compilerOptions: {
                strict: true
            }
        }));
        await reloadPromise;
        async function testAfterReload() {
            const newLs = await (0, service_1.getService)(path_1.default.join(dirPath, 'random.svelte'), rootUris, {
                ...lsDocumentContext,
                watchTsConfig: true
            });
            assert_1.default.strictEqual(newLs.compilerOptions.strict, true, 'expected to reload compilerOptions');
            return true;
        }
    });
    it('can watch project reference tsconfig', async () => {
        const dirPath = (0, test_utils_1.getRandomVirtualDirPath)(testDir);
        const { virtualSystem, lsDocumentContext, rootUris } = setup();
        const tsconfigPath = path_1.default.join(dirPath, 'tsconfig.json');
        const referenced = './tsconfig_node.json';
        const referencedConfigPathFull = path_1.default.resolve(path_1.default.dirname(tsconfigPath), referenced);
        virtualSystem.writeFile(tsconfigPath, JSON.stringify({
            references: [{ path: referenced }],
            include: []
        }));
        virtualSystem.writeFile(referencedConfigPathFull, JSON.stringify({
            compilerOptions: {
                strict: true
            },
            files: ['random.ts']
        }));
        const { reloadPromise, docContextWithReload } = createReloadTester({ ...lsDocumentContext, watchTsConfig: true }, testAfterReload);
        const tsFilePath = path_1.default.join(dirPath, 'random.ts');
        virtualSystem.writeFile(tsFilePath, 'const a: number = null;');
        const ls = await (0, service_1.getService)(tsFilePath, rootUris, docContextWithReload);
        assert_1.default.deepStrictEqual(getSemanticDiagnosticsMessages(ls, tsFilePath), [
            "Type 'null' is not assignable to type 'number'."
        ]);
        virtualSystem.writeFile(referencedConfigPathFull, JSON.stringify({
            compilerOptions: {
                strict: false
            }
        }));
        await reloadPromise;
        async function testAfterReload(reloadingConfigs) {
            if (!reloadingConfigs.includes(referencedConfigPathFull)) {
                return false;
            }
            const newLs = await (0, service_1.getService)(tsFilePath, rootUris, {
                ...lsDocumentContext,
                watchTsConfig: true
            });
            assert_1.default.deepStrictEqual(getSemanticDiagnosticsMessages(newLs, tsFilePath), []);
            return true;
        }
    });
    it('can open client file that do not exist in fs', async () => {
        const dirPath = (0, test_utils_1.getRandomVirtualDirPath)(testDir);
        const { lsDocumentContext, rootUris } = setup();
        // don't need tsconfig because files doesn't exist in fs goes to a service with default config
        const ls = await (0, service_1.getService)(path_1.default.join(dirPath, 'random.svelte'), rootUris, lsDocumentContext);
        const document = new documents_1.Document((0, utils_1.pathToUrl)(path_1.default.join(dirPath, 'random.ts')), '');
        document.openedByClient = true;
        ls.updateSnapshot(document);
        assert_1.default.doesNotThrow(() => {
            ls.getService().getSemanticDiagnostics(document.getFilePath());
        });
    });
    it('resolve module with source project reference redirect', async () => {
        const dirPath = (0, test_utils_1.getRandomVirtualDirPath)(testDir);
        const { virtualSystem, lsDocumentContext, rootUris } = setup();
        const package1 = path_1.default.join(dirPath, 'package1');
        virtualSystem.writeFile(path_1.default.join(package1, 'tsconfig.json'), JSON.stringify({
            references: [{ path: '../package2' }],
            files: ['index.ts']
        }));
        const package2 = path_1.default.join(dirPath, 'package2');
        virtualSystem.writeFile(path_1.default.join(package2, 'tsconfig.json'), JSON.stringify({
            compilerOptions: {
                composite: true,
                strict: true
            },
            files: ['index.ts']
        }));
        const importing = path_1.default.join(package1, 'index.ts');
        virtualSystem.writeFile(importing, 'import { hi } from "package2"; hi((a) => `${a}`);');
        const imported = path_1.default.join(package2, 'index.ts');
        virtualSystem.writeFile(imported, 'export function hi(cb: (num: number) => string) {}');
        const package2Link = (0, utils_1.normalizePath)(path_1.default.join(package1, 'node_modules', 'package2'));
        virtualSystem.realpath = (p) => {
            if ((0, utils_1.normalizePath)(p).startsWith(package2Link)) {
                const sub = p.substring(package2Link.length);
                return path_1.default.join(package2) + sub;
            }
            return p;
        };
        const fileExists = virtualSystem.fileExists;
        virtualSystem.fileExists = (p) => {
            const realPath = virtualSystem.realpath(p);
            return fileExists(realPath);
        };
        const ls = await (0, service_1.getService)(importing, rootUris, lsDocumentContext);
        assert_1.default.deepStrictEqual(getSemanticDiagnosticsMessages(ls, importing), []);
    });
    it('resolve module with source project reference redirect having different module resolution', async () => {
        const dirPath = (0, test_utils_1.getRandomVirtualDirPath)(testDir);
        const { virtualSystem, lsDocumentContext, rootUris } = setup();
        const package1 = path_1.default.join(dirPath, 'package1');
        virtualSystem.writeFile(path_1.default.join(package1, 'tsconfig.json'), JSON.stringify({
            references: [{ path: '../package2' }],
            files: ['index.ts'],
            compilerOptions: {
                moduleResolution: 'Bundler',
                module: 'ESNext',
                paths: {
                    package2: ['../package2']
                }
            }
        }));
        const package2 = path_1.default.join(dirPath, 'package2');
        virtualSystem.writeFile(path_1.default.join(package2, 'tsconfig.json'), JSON.stringify({
            compilerOptions: {
                composite: true,
                strict: true,
                moduleResolution: 'NodeNext'
            },
            files: ['index.ts']
        }));
        const importing = path_1.default.join(package1, 'index.ts');
        virtualSystem.writeFile(importing, 'import { hi } from "package2"; hi((a) => `${a}`);');
        const reExport = path_1.default.join(package2, 'index.ts');
        virtualSystem.writeFile(reExport, 'export * from "./foo"');
        const exportFile = path_1.default.join(package2, 'foo.ts');
        virtualSystem.writeFile(exportFile, 'export function hi(cb: (num: number) => string) {}');
        const ls = await (0, service_1.getService)(importing, rootUris, lsDocumentContext);
        assert_1.default.deepStrictEqual(getSemanticDiagnosticsMessages(ls, importing), []);
    });
    it('skip directory watching if directory is root', async () => {
        const dirPath = (0, test_utils_1.getRandomVirtualDirPath)(path_1.default.join(testDir, 'Test'));
        const { virtualSystem, lsDocumentContext } = setup();
        const rootUris = [(0, utils_1.pathToUrl)(dirPath)];
        const watchDirectory = sinon_1.default.spy();
        lsDocumentContext.watchDirectory = watchDirectory;
        lsDocumentContext.nonRecursiveWatchPattern = '*.ts';
        virtualSystem.readDirectory = () => [];
        virtualSystem.directoryExists = () => true;
        virtualSystem.writeFile(path_1.default.join(dirPath, 'tsconfig.json'), JSON.stringify({
            compilerOptions: {},
            include: ['src/**/*.ts', 'test/**/*.ts', '../foo/**/*.ts']
        }));
        await (0, service_1.getService)(path_1.default.join(dirPath, 'random.svelte'), rootUris, lsDocumentContext);
        sinon_1.default.assert.calledWith(watchDirectory.firstCall, [
            {
                baseUri: (0, utils_1.pathToUrl)(path_1.default.join(dirPath, '../foo')),
                pattern: '**/*.ts'
            }
        ]);
    });
    it('skip directory watching if directory do not exist', async () => {
        const dirPath = (0, test_utils_1.getRandomVirtualDirPath)(path_1.default.join(testDir, 'Test'));
        const { virtualSystem, lsDocumentContext } = setup();
        const rootUris = [(0, utils_1.pathToUrl)(dirPath)];
        const watchDirectory = sinon_1.default.spy();
        lsDocumentContext.watchDirectory = watchDirectory;
        lsDocumentContext.nonRecursiveWatchPattern = '*.ts';
        virtualSystem.readDirectory = () => [];
        virtualSystem.directoryExists = () => false;
        virtualSystem.writeFile(path_1.default.join(dirPath, 'tsconfig.json'), JSON.stringify({
            compilerOptions: {},
            include: ['../test/**/*.ts']
        }));
        await (0, service_1.getService)(path_1.default.join(dirPath, 'random.svelte'), rootUris, lsDocumentContext);
        sinon_1.default.assert.calledWith(watchDirectory.firstCall, []);
    });
    it('assigns files to service with the file in the program', async () => {
        const dirPath = (0, test_utils_1.getRandomVirtualDirPath)(testDir);
        const { virtualSystem, lsDocumentContext, rootUris } = setup();
        const tsconfigPath = path_1.default.join(dirPath, 'tsconfig.json');
        virtualSystem.writeFile(tsconfigPath, JSON.stringify({
            compilerOptions: {
                noImplicitOverride: true
            },
            include: ['src/*.ts']
        }));
        const referencedFile = path_1.default.join(dirPath, 'anotherPackage', 'index.svelte');
        const tsFilePath = path_1.default.join(dirPath, 'src', 'random.ts');
        virtualSystem.readDirectory = () => [tsFilePath];
        virtualSystem.writeFile(referencedFile, '<script lang="ts">class A { a =1 }; class B extends A { a =2 };</script>');
        virtualSystem.writeFile(tsFilePath, 'import "../anotherPackage/index.svelte";');
        const document = new documents_1.Document((0, utils_1.pathToUrl)(referencedFile), virtualSystem.readFile(referencedFile));
        document.openedByClient = true;
        const ls = await (0, service_1.getService)(referencedFile, rootUris, lsDocumentContext);
        ls.updateSnapshot(document);
        assert_1.default.equal((0, utils_1.normalizePath)(ls.tsconfigPath), (0, utils_1.normalizePath)(tsconfigPath));
        const noImplicitOverrideErrorCode = 4114;
        const findError = (ls) => ls
            .getService()
            .getSemanticDiagnostics(referencedFile)
            .find((f) => f.code === noImplicitOverrideErrorCode);
        assert_1.default.ok(findError(ls));
        virtualSystem.writeFile(tsFilePath, '');
        ls.updateTsOrJsFile(tsFilePath);
        const ls2 = await (0, service_1.getService)(referencedFile, rootUris, lsDocumentContext);
        ls2.updateSnapshot(document);
        assert_1.default.deepStrictEqual(findError(ls2), undefined);
    });
    it('assigns newly created files to the right service before the watcher trigger', async () => {
        const dirPath = (0, test_utils_1.getRandomVirtualDirPath)(testDir);
        const { virtualSystem, lsDocumentContext, rootUris } = setup();
        const tsconfigPath = path_1.default.join(dirPath, 'tsconfig.json');
        virtualSystem.writeFile(tsconfigPath, JSON.stringify({
            compilerOptions: {}
        }));
        const svelteFilePath = path_1.default.join(dirPath, 'random.svelte');
        virtualSystem.writeFile(svelteFilePath, '');
        const ls = await (0, service_1.getService)(svelteFilePath, rootUris, lsDocumentContext);
        assert_1.default.equal((0, utils_1.normalizePath)(ls.tsconfigPath), (0, utils_1.normalizePath)(tsconfigPath));
        const svelteFilePath2 = path_1.default.join(dirPath, 'random2.svelte');
        virtualSystem.writeFile(svelteFilePath2, '');
        const ls2 = await (0, service_1.getService)(svelteFilePath2, rootUris, lsDocumentContext);
        assert_1.default.equal((0, utils_1.normalizePath)(ls2.tsconfigPath), (0, utils_1.normalizePath)(tsconfigPath));
    });
    function getSemanticDiagnosticsMessages(ls, filePath) {
        return ls
            .getService()
            .getSemanticDiagnostics(filePath)
            .map((d) => typescript_1.default.flattenDiagnosticMessageText(d.messageText, '\n'));
    }
});
//# sourceMappingURL=service.test.js.map