"use strict";
/*
 * SPDX-License-Identifier: Apache-2.0
 *
 * The OpenSearch Contributors require contributions made to
 * this file be licensed under the Apache-2.0 license or a
 * compatible open source license.
 *
 * Any modifications Copyright OpenSearch Contributors. See
 * GitHub history for details.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.internals = void 0;
const tslib_1 = require("tslib");
/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
const joi_1 = tslib_1.__importDefault(require("joi"));
const lodash_1 = require("lodash");
const moment_1 = require("moment");
const stream_1 = require("stream");
const byte_size_value_1 = require("../byte_size_value");
const duration_1 = require("../duration");
function isMap(o) {
    return o instanceof Map;
}
const anyCustomRule = {
    name: 'custom',
    params: {
        validator: joi_1.default.func().maxArity(1).required(),
    },
    validate(params, value, state, options) {
        let validationResultMessage;
        try {
            validationResultMessage = params.validator(value);
        }
        catch (e) {
            validationResultMessage = e.message || e;
        }
        if (typeof validationResultMessage === 'string') {
            return this.createError('any.custom', { value, message: validationResultMessage }, state, options);
        }
        return value;
    },
};
/**
 * @internal
 */
exports.internals = joi_1.default.extend([
    {
        name: 'any',
        rules: [anyCustomRule],
    },
    {
        name: 'boolean',
        base: joi_1.default.boolean(),
        coerce(value, state, options) {
            // If value isn't defined, let Joi handle default value if it's defined.
            if (value === undefined) {
                return value;
            }
            // Allow strings 'true' and 'false' to be coerced to booleans (case-insensitive).
            // From Joi docs on `Joi.boolean`:
            // > Generates a schema object that matches a boolean data type. Can also
            // >  be called via bool(). If the validation convert option is on
            // > (enabled by default), a string (either "true" or "false") will be
            // converted to a boolean if specified.
            if (typeof value === 'string') {
                const normalized = value.toLowerCase();
                value = normalized === 'true' ? true : normalized === 'false' ? false : value;
            }
            if (typeof value !== 'boolean') {
                return this.createError('boolean.base', { value }, state, options);
            }
            return value;
        },
        rules: [anyCustomRule],
    },
    {
        name: 'binary',
        base: joi_1.default.binary(),
        coerce(value, state, options) {
            // If value isn't defined, let Joi handle default value if it's defined.
            if (value !== undefined && !(typeof value === 'object' && Buffer.isBuffer(value))) {
                return this.createError('binary.base', { value }, state, options);
            }
            return value;
        },
        rules: [anyCustomRule],
    },
    {
        name: 'stream',
        pre(value, state, options) {
            // If value isn't defined, let Joi handle default value if it's defined.
            if (value instanceof stream_1.Stream) {
                return value;
            }
            return this.createError('stream.base', { value }, state, options);
        },
        rules: [anyCustomRule],
    },
    {
        name: 'string',
        base: joi_1.default.string(),
        rules: [anyCustomRule],
    },
    {
        name: 'bytes',
        coerce(value, state, options) {
            try {
                if (typeof value === 'string') {
                    return byte_size_value_1.ByteSizeValue.parse(value);
                }
                if (typeof value === 'number') {
                    return new byte_size_value_1.ByteSizeValue(value);
                }
            }
            catch (e) {
                return this.createError('bytes.parse', { value, message: e.message }, state, options);
            }
            return value;
        },
        pre(value, state, options) {
            // If value isn't defined, let Joi handle default value if it's defined.
            if (value instanceof byte_size_value_1.ByteSizeValue) {
                return value;
            }
            return this.createError('bytes.base', { value }, state, options);
        },
        rules: [
            anyCustomRule,
            {
                name: 'min',
                params: {
                    limit: joi_1.default.alternatives([joi_1.default.number(), joi_1.default.string()]).required(),
                },
                validate(params, value, state, options) {
                    const limit = (0, byte_size_value_1.ensureByteSizeValue)(params.limit);
                    if (value.isLessThan(limit)) {
                        return this.createError('bytes.min', { value, limit }, state, options);
                    }
                    return value;
                },
            },
            {
                name: 'max',
                params: {
                    limit: joi_1.default.alternatives([joi_1.default.number(), joi_1.default.string()]).required(),
                },
                validate(params, value, state, options) {
                    const limit = (0, byte_size_value_1.ensureByteSizeValue)(params.limit);
                    if (value.isGreaterThan(limit)) {
                        return this.createError('bytes.max', { value, limit }, state, options);
                    }
                    return value;
                },
            },
        ],
    },
    {
        name: 'duration',
        coerce(value, state, options) {
            try {
                if (typeof value === 'string' || typeof value === 'number') {
                    return (0, duration_1.ensureDuration)(value);
                }
            }
            catch (e) {
                return this.createError('duration.parse', { value, message: e.message }, state, options);
            }
            return value;
        },
        pre(value, state, options) {
            if (!(0, moment_1.isDuration)(value)) {
                return this.createError('duration.base', { value }, state, options);
            }
            return value;
        },
        rules: [anyCustomRule],
    },
    {
        name: 'number',
        base: joi_1.default.number(),
        coerce(value, state, options) {
            // If value isn't defined, let Joi handle default value if it's defined.
            if (value === undefined) {
                return value;
            }
            // Do we want to allow strings that can be converted, e.g. "2"? (Joi does)
            // (this can for example be nice in http endpoints with query params)
            //
            // From Joi docs on `Joi.number`:
            // > Generates a schema object that matches a number data type (as well as
            // > strings that can be converted to numbers)
            const coercedValue = typeof value === 'string' ? Number(value) : value;
            if (typeof coercedValue !== 'number' || isNaN(coercedValue)) {
                return this.createError('number.base', { value }, state, options);
            }
            return value;
        },
        rules: [anyCustomRule],
    },
    {
        name: 'object',
        base: joi_1.default.object(),
        coerce(value, state, options) {
            if (value === undefined || (0, lodash_1.isPlainObject)(value)) {
                return value;
            }
            if (options.convert && typeof value === 'string') {
                try {
                    const parsed = JSON.parse(value);
                    if ((0, lodash_1.isPlainObject)(parsed)) {
                        return parsed;
                    }
                    return this.createError('object.base', { value: parsed }, state, options);
                }
                catch (e) {
                    return this.createError('object.parse', { value }, state, options);
                }
            }
            return this.createError('object.base', { value }, state, options);
        },
        rules: [anyCustomRule],
    },
    {
        name: 'map',
        coerce(value, state, options) {
            if (value === undefined) {
                return value;
            }
            if ((0, lodash_1.isPlainObject)(value)) {
                return new Map(Object.entries(value));
            }
            if (options.convert && typeof value === 'string') {
                try {
                    const parsed = JSON.parse(value);
                    if ((0, lodash_1.isPlainObject)(parsed)) {
                        return new Map(Object.entries(parsed));
                    }
                    return this.createError('map.base', { value: parsed }, state, options);
                }
                catch (e) {
                    return this.createError('map.parse', { value }, state, options);
                }
            }
            return value;
        },
        pre(value, state, options) {
            if (!isMap(value)) {
                return this.createError('map.base', { value }, state, options);
            }
            return value;
        },
        rules: [
            anyCustomRule,
            {
                name: 'entries',
                params: {
                    key: joi_1.default.object().schema(),
                    value: joi_1.default.object().schema(),
                },
                validate(params, value, state, options) {
                    const result = new Map();
                    for (const [entryKey, entryValue] of value) {
                        const { value: validatedEntryKey, error: keyError } = joi_1.default.validate(entryKey, params.key, { presence: 'required' });
                        if (keyError) {
                            return this.createError('map.key', { entryKey, reason: keyError }, state, options);
                        }
                        const { value: validatedEntryValue, error: valueError } = joi_1.default.validate(entryValue, params.value, { presence: 'required' });
                        if (valueError) {
                            return this.createError('map.value', { entryKey, reason: valueError }, state, options);
                        }
                        result.set(validatedEntryKey, validatedEntryValue);
                    }
                    return result;
                },
            },
        ],
    },
    {
        name: 'record',
        pre(value, state, options) {
            if (value === undefined || (0, lodash_1.isPlainObject)(value)) {
                return value;
            }
            if (options.convert && typeof value === 'string') {
                try {
                    const parsed = JSON.parse(value);
                    if ((0, lodash_1.isPlainObject)(parsed)) {
                        return parsed;
                    }
                    return this.createError('record.base', { value: parsed }, state, options);
                }
                catch (e) {
                    return this.createError('record.parse', { value }, state, options);
                }
            }
            return this.createError('record.base', { value }, state, options);
        },
        rules: [
            anyCustomRule,
            {
                name: 'entries',
                params: { key: joi_1.default.object().schema(), value: joi_1.default.object().schema() },
                validate(params, value, state, options) {
                    const result = {};
                    for (const [entryKey, entryValue] of Object.entries(value)) {
                        const { value: validatedEntryKey, error: keyError } = joi_1.default.validate(entryKey, params.key, { presence: 'required' });
                        if (keyError) {
                            return this.createError('record.key', { entryKey, reason: keyError }, state, options);
                        }
                        const { value: validatedEntryValue, error: valueError } = joi_1.default.validate(entryValue, params.value, { presence: 'required' });
                        if (valueError) {
                            return this.createError('record.value', { entryKey, reason: valueError }, state, options);
                        }
                        result[validatedEntryKey] = validatedEntryValue;
                    }
                    return result;
                },
            },
        ],
    },
    {
        name: 'array',
        base: joi_1.default.array(),
        coerce(value, state, options) {
            if (value === undefined || Array.isArray(value)) {
                return value;
            }
            if (options.convert && typeof value === 'string') {
                try {
                    const parsed = JSON.parse(value);
                    if (Array.isArray(parsed)) {
                        return parsed;
                    }
                    return this.createError('array.base', { value: parsed }, state, options);
                }
                catch (e) {
                    return this.createError('array.parse', { value }, state, options);
                }
            }
            return this.createError('array.base', { value }, state, options);
        },
        rules: [anyCustomRule],
    },
]);
