"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.setIsTrusted = setIsTrusted;
exports.getPackageInfo = getPackageInfo;
exports.importPrettier = importPrettier;
exports.importSvelte = importSvelte;
exports.importSveltePreprocess = importSveltePreprocess;
const path_1 = require("path");
const logger_1 = require("./logger");
/**
 * Whether or not the current workspace can be trusted.
 * TODO rework this to a class which depends on the LsConfigManager
 * and inject that class into all places where it's needed (Document etc.)
 */
let isTrusted = true;
function setIsTrusted(_isTrusted) {
    isTrusted = _isTrusted;
}
/**
 * This function encapsulates the require call in one place
 * so we can replace its content inside rollup builds
 * so it's not transformed.
 */
function dynamicRequire(dynamicFileToRequire) {
    // prettier-ignore
    return require(dynamicFileToRequire);
}
function getPackageInfo(packageName, fromPath, use_fallback = true) {
    const paths = [];
    if (isTrusted) {
        paths.push(fromPath);
    }
    if (use_fallback) {
        paths.push(__dirname);
    }
    const packageJSONPath = require.resolve(`${packageName}/package.json`, {
        paths
    });
    const { version } = dynamicRequire(packageJSONPath);
    const [major, minor, patch] = version.split('.');
    return {
        path: (0, path_1.dirname)(packageJSONPath),
        version: {
            full: version,
            major: Number(major),
            minor: Number(minor),
            patch: Number(patch)
        }
    };
}
function importPrettier(fromPath) {
    const pkg = getPackageInfo('prettier', fromPath);
    const main = (0, path_1.resolve)(pkg.path);
    logger_1.Logger.debug('Using Prettier v' + pkg.version.full, 'from', main);
    return dynamicRequire(main);
}
function importSvelte(fromPath) {
    const pkg = getPackageInfo('svelte', fromPath);
    const main = (0, path_1.resolve)(pkg.path, 'compiler');
    logger_1.Logger.debug('Using Svelte v' + pkg.version.full, 'from', main);
    if (pkg.version.major === 4) {
        return dynamicRequire(main + '.cjs');
    }
    else {
        return dynamicRequire(main);
    }
}
/** Can throw because no fallback guaranteed */
function importSveltePreprocess(fromPath) {
    const pkg = getPackageInfo('svelte-preprocess', fromPath, false // svelte-language-server doesn't have a dependency on svelte-preprocess so we can't provide a fallback
    );
    const main = (0, path_1.resolve)(pkg.path);
    logger_1.Logger.debug('Using svelte-preprocess v' + pkg.version.full, 'from', main);
    return dynamicRequire(main);
}
//# sourceMappingURL=importPackage.js.map